<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * @property CI_DB_query_builder $db
 */
class Dashboard_model extends MY_Model
{
    public function __construct()
    {
        parent::__construct();
    }
    public function sum_selected($column, $user_id = null)
    {
        $this->db->select_sum("nas_router.$column")
            ->from('nas_router')
            ->join('login_credential', 'nas_router.assign_to = login_credential.user_id')
            ->where('login_credential.role', ROLE_CUSTOMER_ID);

        if (in_array(loggedin_role_id(), [ROLE_RESELLER_ID, ROLE_SUBRESELLER_ID])) {
            $this->db->where('nas_router.assign_to', $user_id);
        }

        if ($user_id && loggedin_role_id() == ROLE_ADMIN_ID) {
            $this->db->where('nas_router.assign_to', $user_id);
        }

        $result = $this->db->get()->row_array();

        return (int) $result[$column];
    }
    public function get_monthly_meal_summary($user_id, $month, $year): array
    {
        $this->db->select("
            selected_dates,
            SUM(selected_breakfast) as breakfast,
            SUM(selected_lunch) as lunch,
            SUM(selected_dinner) as dinner
        ")
            ->from('nas_router')
            ->where('assign_to', $user_id)
            ->where('MONTH(selected_dates)', $month)
            ->where('YEAR(selected_dates)', $year)
            ->group_by('selected_dates')
            ->order_by('selected_dates', 'ASC');

        return $this->db->get()->result_array();
    }
    public function get_user_meal_calendar_data($user_id)
    {
        if (!$user_id) return [];

        $this->db->select("
            DATE(selected_dates) as date,
            SUM(selected_breakfast) as breakfast,
            SUM(selected_lunch) as lunch,
            SUM(selected_dinner) as dinner
        ");
        $this->db->from('nas_router');
        $this->db->where('assign_to', $user_id);
        $this->db->where('MONTH(selected_dates)', date('m'));
        $this->db->where('YEAR(selected_dates)', date('Y'));
        $this->db->group_by('DATE(selected_dates)');
        $this->db->order_by('DATE(selected_dates)', 'ASC');

        return $this->db->get()->result_array();
    }

    public function get_client_count_by_status($status)
    {
        if (!in_array($status, [0, 1])) {
            return 0;
        }

        $this->db->where('online', $status);
        $count = $this->db->count_all_results('staff');
        return $count;
    }

    public function get_auto_hold_client_count()

    {

        $this->db->where('auto_hold', 1);
        $count = $this->db->count_all_results('staff');
        return $count;
    }



    public function get_nas_details_by_id($id = '')
    {
        $this->db->select('*');
        $this->db->from('nas_router');
        $this->db->where('id', $id);
        $query = $this->db->get();
        return $query->row_array();
    }

    // get income vs expense chart function
    public function get_income_vs_expense()
    {
        $year = date('Y');
        $cr = array();
        $dr = array();
        for ($month = 1; $month <= 12; $month++) {
            $date = $year . '-' . $month . '-1';
            $month_start = date('Y-m-d', strtotime($date));
            $month_end = date("Y-m-t", strtotime($date));
            $query = 'SELECT IFNULL(SUM(dr),0) as dr, IFNULL(SUM(cr),0) as cr FROM transactions where date between ' . $this->db->escape($month_start) . ' AND ' . $this->db->escape($month_end);
            $query = $this->db->query($query);
            $result = $query->row_array();
            $cr[] = (float)$result['cr'];
            $dr[] = (float)$result['dr'];
        }
        return array('income' => $cr, 'expense' => $dr);
    }

    // get monthly patient fees chart function
    public function get_monthly_patient_fees()
    {
        $days = array();
        $total_bill = array();
        $total_paid = array();
        $total_due = array();
        $startdate = date('Y-m-01');
        $enddate = date('Y-m-t');
        $start = strtotime($startdate);
        $end = strtotime($enddate);
        while ($start <= $end) {
            $today = date('Y-m-d', $start);
            $query = 'SELECT IFNULL(SUM(total-discount+tax_amount),0) as net_bill,IFNULL(SUM(paid),0) as paid,IFNULL(SUM(due),0) as due FROM billing WHERE date = ' . $this->db->escape($today);
            $query = $this->db->query($query);
            $result = $query->row_array();
            $days[] = date('d', $start);
            $total_bill[] = (float)$result['net_bill'];
            $total_paid[] = (float)$result['paid'];
            $total_due[] = (float)$result['due'];
            $start = strtotime('+1 day', $start);
        }
        return array('total_bill' => $total_bill, 'total_paid' => $total_paid, 'total_due' => $total_due, 'days' => $days);
    }


    public function get_monthly_bill_totals($user_id = null)
    {
        $currentYear = date('Y');
        $currentMonth = date('m');

        // Get the total bills for each day in the current year
        $this->db->select('DAY(billing.date) AS day, MONTH(billing.date) AS month, IFNULL(SUM(total - discount + tax_amount), 0) AS total_bill');
        $this->db->join('staff', 'staff.id = billing.staff_id');
        $this->db->where('YEAR(billing.date)', $currentYear); // Filter by the current year

        if ($user_id) {
            $this->db->where('staff.assign_to', $user_id); // Filter by user if provided
        }

        // Group by both day and month
        $this->db->group_by('DAY(billing.date), MONTH(billing.date)');
        $this->db->order_by('MONTH(billing.date), DAY(billing.date)', 'ASC');

        $query = $this->db->get('billing')->result_array();

        $monthly_bills = [];

        // Initialize all days of the month with 0
        for ($i = 1; $i <= 31; $i++) {
            $monthly_bills[$currentMonth][$i] = 0; // Default value
        }

        // Populate the monthly bills with the data for each day
        foreach ($query as $row) {
            $monthly_bills[$row['month']][$row['day']] = (float)$row['total_bill'];
        }

        // Return the structured data for the month with all days filled in
        return $monthly_bills;
    }


    // get today invoice quantity function
    public function get_today_invoice_qty()
    {
        $query = $this->db->select('count(id) as total_invoice')
            ->where('date', date('Y-m-d'))
            ->get('billing')->row_array();
        return $query['total_invoice'];
    }

    // get today commission
    public function get_today_commission()
    {
        $query = $this->db->select('IFNULL(SUM(commission),0) as total_commission')
            ->where('date', date('Y-m-d'))
            ->get('billing')->row_array();
        return $query['total_commission'];
    }

    // get today income and expense amount
    public function get_today_incomeandexpense()
    {
        $query = $this->db->select('IFNULL(SUM(dr),0) as total_expense,IFNULL(SUM(cr),0) as total_income')
            ->where('date', date('Y-m-d'))
            ->get('transactions')->row_array();
        return $query;
    }

    // get total patient
    public function get_total_client()
    {
        $query = $this->db->select('count(id) as total_client')
            ->get('staff')->row_array();
        return $query['total_client'];
    }

    // get total doctor
    public function get_total_doctor()
    {
        $query = $this->db->select('count(id) as total_doctor')
            ->where('role', 3)
            ->get('login_credential')->row_array();
        return $query['total_doctor'];
    }

    // get total staff
    public function get_total_staff(array $roles = [1, 3, 7], $assign_to = null)
    {
        $this->db->select('COUNT(login_credential.id) as total_staff')
            ->from('login_credential')
            ->join('roles', 'roles.id = login_credential.role', 'inner')
            ->where_in('login_credential.role', $roles);



        $query = $this->db->get()->row_array();
        return $query['total_staff'];
    }


    // get total appointment no
    public function get_total_appointment()
    {
        $query = $this->db->select('count(id) as total_appointment')
            ->where('status', 4)
            ->get('appointment')->row_array();
        return $query['total_appointment'];
    }

    // get total appointment no
    public function get_monthly_appointment()
    {
        $confirmed = 0;
        $canceled = 0;
        $pending = 0;
        $this->db->select('id,status');
        $this->db->from('appointment');
        $this->db->where('MONTH(appointment_date)', date('m'));
        $this->db->where('YEAR(appointment_date)', date('Y'));
        $result = $this->db->get()->result_array();
        foreach ($result as $key => $value) {
            if ($value['status'] == 1 || $value['status'] == 2) {
                $confirmed++;
            }
            if ($value['status'] == 3) {
                $canceled++;
            }
            if ($value['status'] == 4) {
                $pending++;
            }
        }
        $array = array($confirmed, $pending, $canceled);
        return $array;
    }

    public function get_clients_status($user_id = null): ?array
    {
        $this->db->select('IFNULL(SUM(staff.status="Active"), 0) as Active')
            ->select('IFNULL(SUM(staff.status="Inactive"), 0) as Inactive')
            ->select('IFNULL(SUM(staff.status="Hold"), 0) as Hold')
            ->from('staff')
            ->join('login_credential', 'staff.id = login_credential.user_id')
            ->where('login_credential.role', ROLE_CUSTOMER_ID);
    
        // Only resellers and sub-resellers should be restricted by assign_to
        if (in_array(loggedin_role_id(), [ROLE_RESELLER_ID, ROLE_SUBRESELLER_ID])) {
            $this->db->where('staff.assign_to', $user_id);
        }
    
        // ROLE_ADMIN_ID (role_id = 1) can see all, so skip filtering by assign_to
        // Nothing needed here for admin since they should see all
    
        return $this->db->get()->row_array();
    }
    
    public function get_monthly_new_clients_graph_data($user_id = null): array
    {
        // Prepare the result array
        $monthly_data = [];

        // Loop through each month of the current year
        for ($month = 1; $month <= 12; $month++) {
            $this->db->select('COUNT(*) as TotalClients')
                ->from('staff')
                ->join('login_credential', 'staff.id = login_credential.user_id')
                ->where('login_credential.role', ROLE_CUSTOMER_ID)
                ->where('MONTH(staff.created_at)', $month)
                ->where('YEAR(staff.created_at)', date('Y'));

            // Apply filters based on the role
            if (in_array(loggedin_role_id(), [ROLE_RESELLER_ID, ROLE_SUBRESELLER_ID])) {
                $this->db->where('staff.assign_to', $user_id);
            }

            if ($user_id && loggedin_role_id() == ROLE_ADMIN_ID) {
                $this->db->where('staff.assign_to', $user_id);
            }

            // Execute the query and add the result to the monthly data
            $result = $this->db->get()->row_array();
            $monthly_data[] = [
                'month' => $month,
                'TotalClients' => (int) $result['TotalClients']
            ];
        }

        return $monthly_data;
    }
    public function get_monthly_new_clients_count($user_id = null): ?array
    {
        $this->db->select('COUNT(*) as NewClients')
            ->from('staff')
            ->join('login_credential', 'staff.id = login_credential.user_id')
            ->where('login_credential.role', ROLE_CUSTOMER_ID);

        $role_id = loggedin_role_id();

        if (in_array($role_id, [ROLE_RESELLER_ID, ROLE_SUBRESELLER_ID])) {
            $this->db->where('staff.assign_to', $user_id);
        }

        // For other roles except admin, apply assign_to condition
        if ($role_id != ROLE_ADMIN_ID && $user_id) {
            $this->db->where('staff.assign_to', $user_id);
        }

        // Filter by current month and year
        $this->db->where('MONTH(staff.created_at)', date('m'));
        $this->db->where('YEAR(staff.created_at)', date('Y'));

        return $this->db->get()->row_array();
    }
    public function get_staff_basic_info($staff_id)
    {
        $this->db->select('
        s.name, 
        s.address,
          s.merchant_no,  
        s.payment_account,
        s.idd, 
        s.photo, 
        s.mobileno,
        s.assign_to, 
        s.status,
        a.name as branch_name,
        a.merchant_no as branch_id,
        a.payment_account as branch_payment_account 
    ');
        $this->db->from('staff as s');
        $this->db->join('staff as a', 'a.id = s.assign_to', 'left');
        $this->db->where('s.id', $staff_id);

        $result = $this->db->get()->row_array();

        if ($result) {
            $result['photo_url'] = base_url('uploads/staff/' . ($result['photo'] ?: 'default.png'));
        } else {
            $result = ['error' => 'Staff not found'];
        }

        return $result;
    }

    public function get_total_refund_count($user_id = null): ?array
    {
        if ($user_id === null) {
            return ['TotalRefundCount' => 0];
        }

        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();

        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            $this->db->select('SUM(refund.selected_breakfast + refund.selected_lunch + refund.selected_dinner) as TotalRefundCount');
            $this->db->from('refund');
            $this->db->where('MONTH(refunded_at)', date('m'));
            $this->db->where('YEAR(refunded_at)', date('Y'));
            $this->db->where('refund.status', 1);
            return $this->db->get()->row_array();
        }

        if (in_array($role, [2, 3])) {
            $this->db->select('SUM(refund.selected_breakfast + refund.selected_lunch + refund.selected_dinner) as TotalRefundCount');
            $this->db->from('refund');
            $this->db->join('staff', 'staff.id = refund.refunded_by', 'left');
            $this->db->group_start();
            $this->db->where('staff.assign_to', $user_id);
            $this->db->or_where('refund.refunded_by', $user_id);
            $this->db->group_end();
            $this->db->where('MONTH(refund.refunded_at)', date('m'));
            $this->db->where('YEAR(refund.refunded_at)', date('Y'));
            $this->db->where('refund.status', 1);

            return $this->db->get()->row_array();
        }

        $this->db->select('SUM(refund.selected_breakfast + refund.selected_lunch + refund.selected_dinner) as TotalRefundCount');
        $this->db->from('refund');
        $this->db->where('refunded_by', $user_id);
        $this->db->where('MONTH(refunded_at)', date('m'));
        $this->db->where('YEAR(refunded_at)', date('Y'));
        $this->db->where('refund.status', 1);

        return $this->db->get()->row_array();
    }

    public function get_refund_count($user_id = null): ?array
    {
        if ($user_id === null) {
            return ['breakfast' => 0, 'lunch' => 0, 'dinner' => 0];
        }


        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();
        $role = $role_data['role'] ?? null;

        $select = [
            'SUM(refund.selected_breakfast) as breakfast',
            'SUM(refund.selected_lunch) as lunch',
            'SUM(refund.selected_dinner) as dinner'
        ];

        $this->db->select($select);
        $this->db->from('refund');
        $this->db->where('MONTH(refund.refunded_at)', date('m'));
        $this->db->where('YEAR(refund.refunded_at)', date('Y'));
        $this->db->where('refund.status', 1);

        if ($role == 1) {
        } elseif (in_array($role, [2, 3])) {
            $this->db->join('staff', 'staff.id = refund.refunded_by', 'left');
            $this->db->group_start();
            $this->db->where('staff.assign_to', $user_id);
            $this->db->or_where('refund.refunded_by', $user_id);
            $this->db->group_end();
        } else {
            $this->db->where('refund.refunded_by', $user_id);
        }

        return $this->db->get()->row_array();
    }
    public function get_refund_costs($user_id = null): array
    {
        $result = ['breakfast' => 0, 'lunch' => 0, 'dinner' => 0];

        if ($user_id === null) return $result;

        // Get user role
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();
        $role = $role_data['role'] ?? null;

        // Base refund + package query
        $this->db->select('
        refund.selected_breakfast,
        refund.selected_lunch,
        refund.selected_dinner,
        packages.breakfast_cost,
        packages.lunch_cost,
        packages.dinner_cost
    ');
        $this->db->from('refund');
        $this->db->join('packages', 'packages.name = refund.package_name', 'left');
        $this->db->where('MONTH(refund.refunded_at)', date('m'));
        $this->db->where('YEAR(refund.refunded_at)', date('Y'));
        $this->db->where('refund.status', 1); // Only approved refunds

        // Role-based filtering
        if ($role == 1) {
            // Admin – see all
        } elseif ($role == 2) {
            // Manager – only clients assigned to them
            $this->db->join('staff', 'staff.id = refund.refunded_by', 'left');
            $this->db->where('staff.assign_to', $user_id);
        } elseif ($role == 3) {
            // Sub-manager – own + assigned
            $this->db->join('staff', 'staff.id = refund.refunded_by', 'left');
            $this->db->group_start();
            $this->db->where('staff.assign_to', $user_id);
            $this->db->or_where('refund.refunded_by', $user_id);
            $this->db->group_end();
        } else {
            // Other users – only their own refunds
            $this->db->where('refund.refunded_by', $user_id);
        }

        $rows = $this->db->get()->result_array();

        foreach ($rows as $row) {
            $result['breakfast'] += (float)$row['selected_breakfast'] * (float)$row['breakfast_cost'];
            $result['lunch']     += (float)$row['selected_lunch'] * (float)$row['lunch_cost'];
            $result['dinner']    += (float)$row['selected_dinner'] * (float)$row['dinner_cost'];
        }

        return [
            'breakfast' => round($result['breakfast'], 2),
            'lunch'     => round($result['lunch'], 2),
            'dinner'    => round($result['dinner'], 2),
        ];
    }


    public function get_daily_meal_summary($user_id)
    {
        if (!$user_id) return [];

        $role = $this->db
            ->where('user_id', $user_id)
            ->get('login_credential')
            ->row('role');

        $staff_ids = [];
        if (in_array($role, [2, 3])) {
            $this->db->select('id');
            $this->db->from('staff');
            $this->db->where('created_by', $user_id);
            $result = $this->db->get()->result_array();
            $staff_ids = array_column($result, 'id');
        }

        $this->db->select("
            DATE(nas_router.selected_dates) as date,
            SUM(selected_breakfast) as breakfast,
            SUM(selected_lunch) as lunch,
            SUM(selected_dinner) as dinner
        ");
        $this->db->from('nas_router');

        if ($role == 1) {
        } elseif (in_array($role, [2, 3])) {
            $this->db->group_start();
            if (!empty($staff_ids)) {
                $this->db->where_in('nas_router.assign_to', $staff_ids);
            }
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
        } else {
            $this->db->where('nas_router.assign_to', $user_id);
        }

        $this->db->where('nas_router.selected_dates IS NOT NULL');
        $this->db->group_by('DATE(nas_router.selected_dates)');
        $this->db->order_by('DATE(nas_router.selected_dates)', 'ASC');

        return $this->db->get()->result_array();
    }
    public function get_total_branch_active($user_id = null)
    {
        if ($user_id === null) {
            return 0;
        }

        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();
        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            $this->db->from('staff');
            $this->db->join('login_credential', 'login_credential.user_id = staff.id');
            $this->db->where('login_credential.role', 2);
            $this->db->where('staff.status', 'active');
            return $this->db->count_all_results();
        }

        $this->db->select('id');
        $this->db->from('staff');
        $this->db->where('assign_to', $user_id);
        $assigned_staff = $this->db->get()->result_array();

        $staff_ids = array_column($assigned_staff, 'id');
        $staff_ids[] = $user_id;

        if (empty($staff_ids)) {
            return 0;
        }

        $this->db->from('staff');
        $this->db->join('login_credential', 'login_credential.user_id = staff.id');
        $this->db->where_in('staff.assign_to', $staff_ids);
        $this->db->where('login_credential.role', 2);
        $this->db->where('staff.status', 'active');
        return $this->db->count_all_results();
    }
    public function get_total_branch_inactive($user_id = null)
    {
        if ($user_id === null) {
            return 0;
        }

        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();
        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            $this->db->from('staff');
            $this->db->join('login_credential', 'login_credential.user_id = staff.id');
            $this->db->where('login_credential.role', 2);
            $this->db->where('staff.status', 'inactive');
            return $this->db->count_all_results();
        }

        $this->db->select('id');
        $this->db->from('staff');
        $this->db->where('assign_to', $user_id);
        $assigned_staff = $this->db->get()->result_array();

        $staff_ids = array_column($assigned_staff, 'id');
        $staff_ids[] = $user_id;

        if (empty($staff_ids)) {
            return 0;
        }

        $this->db->from('staff');
        $this->db->join('login_credential', 'login_credential.user_id = staff.id');
        $this->db->where_in('staff.assign_to', $staff_ids);
        $this->db->where('login_credential.role', 2);
        $this->db->where('staff.status', 'inactive');
        return $this->db->count_all_results();
    }
    public function get_total_branch_hold($user_id = null)
    {
        if ($user_id === null) {
            return 0;
        }

        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();
        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            $this->db->from('staff');
            $this->db->join('login_credential', 'login_credential.user_id = staff.id');
            $this->db->where('login_credential.role', 2);
            $this->db->where('staff.status', 'hold');
            return $this->db->count_all_results();
        }

        $this->db->select('id');
        $this->db->from('staff');
        $this->db->where('assign_to', $user_id);
        $assigned_staff = $this->db->get()->result_array();

        $staff_ids = array_column($assigned_staff, 'id');
        $staff_ids[] = $user_id;

        if (empty($staff_ids)) {
            return 0;
        }

        $this->db->from('staff');
        $this->db->join('login_credential', 'login_credential.user_id = staff.id');
        $this->db->where_in('staff.assign_to', $staff_ids);
        $this->db->where('login_credential.role', 2);
        $this->db->where('staff.status', 'hold');
        return $this->db->count_all_results();
    }

    public function get_total_students($user_id = null)
    {
        if ($user_id === null) {
            return 0;
        }

        // Get user role
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();

        $role = $role_data['role'] ?? null;

        // Admin: count all role 4 users from staff
        if ($role == 1) {
            $this->db->from('staff');
            $this->db->join('login_credential', 'login_credential.user_id = staff.id');
            $this->db->where('login_credential.role', 4);
            return $this->db->count_all_results();
        }

        // Manager or other: get assigned staff IDs
        $this->db->select('id');
        $this->db->from('staff');
        $this->db->where('assign_to', $user_id);
        $assigned_staff = $this->db->get()->result_array();

        $staff_ids = array_column($assigned_staff, 'id');
        $staff_ids[] = $user_id;

        if (empty($staff_ids)) {
            return 0;
        }

        $this->db->from('staff');
        $this->db->join('login_credential', 'login_credential.user_id = staff.id');
        $this->db->where_in('staff.assign_to', $staff_ids);
        $this->db->where('login_credential.role', 4); // Only students
        return $this->db->count_all_results();
    }

    public function get_total_active($user_id = null)
    {
        if ($user_id === null) {
            return 0;
        }

        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();

        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            // Superadmin: count only role 4 and active
            $this->db->from('staff');
            $this->db->join('login_credential', 'login_credential.user_id = staff.id');
            $this->db->where('staff.status', 'active');
            $this->db->where('login_credential.role', 4);
            return $this->db->count_all_results();
        }

        // For others, get assigned staff
        $this->db->select('id');
        $this->db->from('staff');
        $this->db->where('assign_to', $user_id);
        $assigned_staff = $this->db->get()->result_array();

        $staff_ids = array_column($assigned_staff, 'id');
        $staff_ids[] = $user_id;

        if (empty($staff_ids)) {
            return 0;
        }

        $this->db->from('staff');
        $this->db->join('login_credential', 'login_credential.user_id = staff.id');
        $this->db->where_in('staff.assign_to', $staff_ids);
        $this->db->where('staff.status', 'active');
        $this->db->where('login_credential.role', 4); // Only students
        return $this->db->count_all_results();
    }


    public function get_total_inactive($user_id = null)
    {
        if ($user_id === null) {
            return 0;
        }

        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();

        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            // Superadmin: count only role 4 and active
            $this->db->from('staff');
            $this->db->join('login_credential', 'login_credential.user_id = staff.id');
            $this->db->where('staff.status', 'inactive');
            $this->db->where('login_credential.role', 4);
            return $this->db->count_all_results();
        }

        // For others, get assigned staff
        $this->db->select('id');
        $this->db->from('staff');
        $this->db->where('assign_to', $user_id);
        $assigned_staff = $this->db->get()->result_array();

        $staff_ids = array_column($assigned_staff, 'id');
        $staff_ids[] = $user_id;

        if (empty($staff_ids)) {
            return 0;
        }

        $this->db->from('staff');
        $this->db->join('login_credential', 'login_credential.user_id = staff.id');
        $this->db->where_in('staff.assign_to', $staff_ids);
        $this->db->where('staff.status', 'inactive');
        $this->db->where('login_credential.role', 4); // Only students
        return $this->db->count_all_results();
    }
    public function get_pending_meal($user_id)
    {
        if ($user_id === null) {
            return 0;
        }

        $today = date('Y-m-d');
        $month = date('m');
        $year = date('Y');

        // Get user role
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();
        $role = $role_data['role'] ?? null;

        $total_meals = 0;
        $served_meals = 0;

        if ($role == 1) {
            // Superadmin: All meals
            $this->db->select('SUM(selected_breakfast + selected_lunch + selected_dinner) AS total_meals');
            $this->db->from('nas_router');
            $this->db->where('MONTH(created_at)', $month);
            $this->db->where('YEAR(created_at)', $year);
            $total_meals = (int) $this->db->get()->row('total_meals');

            $this->db->select('COUNT(*) AS served');
            $this->db->from('nas_router');
            $this->db->where('selected_dates <=', $today);
            $served_meals = (int) $this->db->get()->row('served');
        } elseif ($role == 2) {
            // Admin: Own meals + meals of staff they created
            $this->db->select('SUM(selected_breakfast + selected_lunch + selected_dinner) AS total_meals');
            $this->db->from('nas_router');
            $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
            $this->db->group_start();
            $this->db->where('staff.created_by', $user_id);
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
            $this->db->where('MONTH(nas_router.created_at)', $month);
            $this->db->where('YEAR(nas_router.created_at)', $year);
            $total_meals = (int) $this->db->get()->row('total_meals');

            $this->db->select('COUNT(*) AS served');
            $this->db->from('nas_router');
            $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
            $this->db->group_start();
            $this->db->where('staff.created_by', $user_id);
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
            $this->db->where('nas_router.selected_dates <=', $today);
            $served_meals = (int) $this->db->get()->row('served');
        } else {
            // Staff (role 4 or other): Only own meals
            $this->db->select('SUM(selected_breakfast + selected_lunch + selected_dinner) AS total_meals');
            $this->db->from('nas_router');
            $this->db->where('assign_to', $user_id);
            $this->db->where('MONTH(created_at)', $month);
            $this->db->where('YEAR(created_at)', $year);
            $total_meals = (int) $this->db->get()->row('total_meals');

            $this->db->select('COUNT(*) AS served');
            $this->db->from('nas_router');
            $this->db->where('assign_to', $user_id);
            $this->db->where('selected_dates <=', $today);
            $served_meals = (int) $this->db->get()->row('served');
        }

        return max($total_meals - $served_meals, 0);
    }
public function get_pending_cost_meal($user_id): array
{
    if ($user_id === null) {
        return [
            'breakfast' => 0,
            'lunch' => 0,
            'dinner' => 0,
        ];
    }

    $today = date('Y-m-d');
    $month = date('m');
    $year = date('Y');

    // Get user role
    $this->db->select('role');
    $this->db->from('login_credential');
    $this->db->where('user_id', $user_id);
    $role_data = $this->db->get()->row_array();
    $role = $role_data['role'] ?? null;

    // Start query from nas_router
    $this->db->from('nas_router');
    $this->db->where('MONTH(nas_router.created_at)', $month);
    $this->db->where('YEAR(nas_router.created_at)', $year);

    // Role-based filtering
    if ($role == 1) {
        // Admin – no restriction
    } elseif (in_array($role, [2, 3])) {
        // Manager / Team Lead – see own and assigned clients
        $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
        $this->db->group_start();
        $this->db->where('staff.assign_to', $user_id); // clients assigned to this manager
        $this->db->or_where('nas_router.assign_to', $user_id); // own
        $this->db->group_end();
    } else {
        // Regular user – only own assigned data
        $this->db->where('nas_router.assign_to', $user_id);
    }

    // Clone query
    $base_query = clone $this->db;

    // Pending Breakfast
    $this->db = clone $base_query;
    $this->db->select('SUM(selected_breakfast) as pending_breakfast');
    $this->db->where('selected_dates >', $today);
    $pending_breakfast = (int) $this->db->get()->row('pending_breakfast');

    // Pending Lunch
    $this->db = clone $base_query;
    $this->db->select('SUM(selected_lunch) as pending_lunch');
    $this->db->where('selected_dates >', $today);
    $pending_lunch = (int) $this->db->get()->row('pending_lunch');

    // Pending Dinner
    $this->db = clone $base_query;
    $this->db->select('SUM(selected_dinner) as pending_dinner');
    $this->db->where('selected_dates >', $today);
    $pending_dinner = (int) $this->db->get()->row('pending_dinner');

    return [
        'breakfast' => $pending_breakfast,
        'lunch' => $pending_lunch,
        'dinner' => $pending_dinner,
    ];
}

    public function get_pending_cost($user_id): array
    {
        $result = ['breakfast' => 0, 'lunch' => 0, 'dinner' => 0];

        if ($user_id === null) {
            return $result;
        }

        $today = date('Y-m-d');
        $month = date('m');
        $year = date('Y');

        // Get role
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();
        $role = $role_data['role'] ?? null;

        // Base Query
        $this->db->select("
        nas_router.selected_breakfast,
        nas_router.selected_lunch,
        nas_router.selected_dinner,
        packages.breakfast_cost,
        packages.lunch_cost,
        packages.dinner_cost
    ");
        $this->db->from('nas_router');
        $this->db->join('packages', 'packages.name = nas_router.package_name', 'left');
        $this->db->where('DATE(nas_router.selected_dates) >', $today);
        $this->db->where('MONTH(nas_router.created_at)', $month);
        $this->db->where('YEAR(nas_router.created_at)', $year);

        // Role-based filter
        if ($role == 1) {
            // Superadmin: see all
        } elseif ($role == 2) {
            $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
            $this->db->group_start();
            $this->db->where('staff.created_by', $user_id);
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
        } else {
            $this->db->where('nas_router.assign_to', $user_id);
        }

        $rows = $this->db->get()->result_array();

        foreach ($rows as $row) {
            $result['breakfast'] += (float) $row['selected_breakfast'] * (float) $row['breakfast_cost'];
            $result['lunch']     += (float) $row['selected_lunch'] * (float) $row['lunch_cost'];
            $result['dinner']    += (float) $row['selected_dinner'] * (float) $row['dinner_cost'];
        }

        return [
            'breakfast' => round($result['breakfast'], 2),
            'lunch'     => round($result['lunch'], 2),
            'dinner'    => round($result['dinner'], 2),
        ];
    }

    public function get_total_selected_breakfast($user_id = null): ?array
    {
        if ($user_id === null) {
            return ['TotalSelectedBreakfast' => 0];
        }

        // Get role
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();

        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            // Superadmin: total of all users
            $this->db->select('SUM(selected_breakfast) as TotalSelectedBreakfast');
            $this->db->from('nas_router');
            $this->db->where('MONTH(created_at)', date('m'));
            $this->db->where('YEAR(created_at)', date('Y'));
            return $this->db->get()->row_array();
        }

        if (in_array($role, [2, 3])) {
            // Manager or Role 3: Include staff + own breakfast
            $this->db->select('SUM(nas_router.selected_breakfast) as TotalSelectedBreakfast');
            $this->db->from('nas_router');
            $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
            $this->db->group_start();
            $this->db->where('staff.created_by', $user_id);
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
            $this->db->where('MONTH(nas_router.created_at)', date('m'));
            $this->db->where('YEAR(nas_router.created_at)', date('Y'));

            return $this->db->get()->row_array();
        }

        // Regular user
        $this->db->select('SUM(selected_breakfast) as TotalSelectedBreakfast');
        $this->db->from('nas_router');
        $this->db->where('assign_to', $user_id);
        $this->db->where('MONTH(created_at)', date('m'));
        $this->db->where('YEAR(created_at)', date('Y'));

        return $this->db->get()->row_array();
    }
    public function get_student_balance($user_id)
    {
        if ((int)$user_id === 1) {
            // Superadmin: sum all student balances
            $this->db->select('SUM(total_amount) as ordercost');
            $this->db->from('nas_router');
            $this->db->where('MONTH(created_at)', date('m'));
            $this->db->where('YEAR(created_at)', date('Y'));

            $row = $this->db->get()->row_array();
            return (int)($row['ordercost'] ?? 0);
        }

        // For other users: Get staff assigned to this user
        $this->db->select('id');
        $this->db->from('staff');
        $this->db->where('assign_to', $user_id);
        $staff_ids = $this->db->get()->result_array();

        if (empty($staff_ids)) {
            return 0;
        }

        // Extract IDs into a flat array
        $ids = array_column($staff_ids, 'id');

        // Now sum total_amount in nas_router for these staff
        $this->db->select('SUM(total_amount) as ordercost');
        $this->db->from('nas_router');
        $this->db->where_in('assign_to', $ids);
        $this->db->where('MONTH(created_at)', date('m'));
        $this->db->where('YEAR(created_at)', date('Y'));

        $row = $this->db->get()->row_array();
        return (int)($row['ordercost'] ?? 0);
    }


    public function get_total_balance($user_id)
    {
        // Get role of the user
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_row = $this->db->get()->row_array();
        $role = $role_row['role'] ?? null;

        // If superadmin, get sum of staff balances who are role 3
        if ((int)$role === 1) {
            $this->db->select('SUM(staff.user_balance) as TotalBalance');
            $this->db->from('staff');
            $this->db->join('login_credential', 'login_credential.user_id = staff.id');
            $this->db->where('login_credential.role', 4);

            $row = $this->db->get()->row_array();
            return (int)($row['TotalBalance'] ?? 0);
        }

        // If role 3, return only their own balance
        if ((int)$role === 2) {
            $this->db->select('SUM(user_balance) as TotalBalance');
            $this->db->from('staff');
            $this->db->where('assign_to', $user_id);

            $row = $this->db->get()->row_array();
            return (int)($row['TotalBalance'] ?? 0);
        }
    }

    public function get_current_total_balance($user_id)
    {
        $this->db->select('SUM(user_balance) as TotalBalance');
        $this->db->from('staff');

        if ((int)$user_id !== 1) {
            $this->db->where('assign_to', $user_id);
        }

        $result = $this->db->get()->row_array();
        return (int)($result['TotalBalance'] ?? 0);
    }

    public function get_credit_in_this_month($user_id)
    {
        $current = $this->get_current_total_balance($user_id);
        $start = $this->get_monthly_credit_in($user_id);

        return $current - $start;
    }
    public function get_total_meal_income($user_id = null): ?array
    {
        if ($user_id === null) {
            return ['TotalMealIncome' => 0];
        }
    
        // Get role of the user
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();
    
        $role = $role_data['role'] ?? null;
    
        if ($role == 1) {
            // Superadmin: sum all users' income
            $this->db->select('SUM(income_amount) as TotalMealIncome');
            $this->db->from('nas_router');
            $this->db->where('MONTH(created_at)', date('m'));
            $this->db->where('YEAR(created_at)', date('Y'));
            return $this->db->get()->row_array();
        }
    
        if (in_array($role, [2, 3])) {
            // Manager or Role 3: own + created staff
            $this->db->select('SUM(nas_router.income_amount) as TotalMealIncome');
            $this->db->from('nas_router');
            $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
            $this->db->group_start();
            $this->db->where('staff.created_by', $user_id);
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
            $this->db->where('MONTH(nas_router.created_at)', date('m'));
            $this->db->where('YEAR(nas_router.created_at)', date('Y'));
            return $this->db->get()->row_array();
        }
    
        // Regular user: only their own income
        $this->db->select('SUM(income_amount) as TotalMealIncome');
        $this->db->from('nas_router');
        $this->db->where('assign_to', $user_id);
        $this->db->where('MONTH(created_at)', date('m'));
        $this->db->where('YEAR(created_at)', date('Y'));
        return $this->db->get()->row_array();
    }
    
    public function get_this_month_expense_total($user_id)
    {
        $this->db->select('IFNULL(SUM(price), 0) as total_expense');
        $this->db->from('supplier');
        $this->db->where('MONTH(date)', date('m'));
        $this->db->where('YEAR(date)', date('Y'));

        // Superadmin (Role 1) can see all

        if ((int)$user_id !== 1) {
            $this->db->where('prepared_by', $user_id);
        }

        $result = $this->db->get()->row_array();
        return $result['total_expense'] ?? 0;
    }


    public function get_total_refund($user_id = null): ?array
    {
        if ($user_id === null) {
            return ['TotalRefund' => 0];
        }

        // Get role
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();

        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            $this->db->select('SUM(total_amount) as TotalRefund');
            $this->db->from('refund');
            $this->db->where('MONTH(refunded_at)', date('m'));
            $this->db->where('YEAR(refunded_at)', date('Y'));
            $this->db->where('refund.status', 1); // Only count status = 1
            return $this->db->get()->row_array();
        }

        if (in_array($role, [2, 3])) {
            $this->db->select('SUM(total_amount) as TotalRefund');
            $this->db->from('refund');
            $this->db->join('staff', 'staff.id = refund.refunded_by', 'left');
            $this->db->group_start();
            $this->db->where('staff.assign_to', $user_id);
            $this->db->or_where('refund.refunded_by', $user_id);
            $this->db->group_end();
            $this->db->where('MONTH(refund.refunded_at)', date('m'));
            $this->db->where('YEAR(refund.refunded_at)', date('Y'));
            $this->db->where('refund.status', 1); // Only count status = 1
            return $this->db->get()->row_array();
        }

        // Regular user
        $this->db->select('SUM(total_amount) as TotalRefund');
        $this->db->from('refund');
        $this->db->where('refunded_by', $user_id);
        $this->db->where('MONTH(refunded_at)', date('m'));
        $this->db->where('YEAR(refunded_at)', date('Y'));
        $this->db->where('refund.status', 1); // Only count status = 1

        return $this->db->get()->row_array();
    }


    public function get_total_selected_lunch_by_user($user_id): ?array
    {
        // Get role
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();

        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            // Superadmin: All users
            $this->db->select('SUM(selected_lunch) as TotalSelectedLunch');
            $this->db->from('nas_router');
            $this->db->where('MONTH(created_at)', date('m'));
            $this->db->where('YEAR(created_at)', date('Y'));
            return $this->db->get()->row_array();
        }

        if (in_array($role, [2, 3])) {
            // Manager or Role 3: Own + staff lunch
            $this->db->select('SUM(nas_router.selected_lunch) as TotalSelectedLunch');
            $this->db->from('nas_router');
            $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
            $this->db->group_start();
            $this->db->where('staff.created_by', $user_id);
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
            $this->db->where('MONTH(nas_router.created_at)', date('m'));
            $this->db->where('YEAR(nas_router.created_at)', date('Y'));

            return $this->db->get()->row_array();
        }

        // Regular user
        $this->db->select('SUM(selected_lunch) as TotalSelectedLunch');
        $this->db->from('nas_router');
        $this->db->where('assign_to', $user_id);
        $this->db->where('MONTH(created_at)', date('m'));
        $this->db->where('YEAR(created_at)', date('Y'));

        return $this->db->get()->row_array();
    }

    public function get_total_selected_dinner($user_id = null): ?array
    {
        if ($user_id === null) {
            return ['TotalSelectedDinner' => 0];
        }

        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();

        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            // Superadmin: See total of all dinners
            $this->db->select('SUM(selected_dinner) as TotalSelectedDinner');
            $this->db->from('nas_router');
            $this->db->where('MONTH(created_at)', date('m'));
            $this->db->where('YEAR(created_at)', date('Y'));
            return $this->db->get()->row_array();
        }

        if (in_array($role, [2, 3])) {
            // Manager or Role 3: Include staff + own dinner
            $this->db->select('SUM(nas_router.selected_dinner) as TotalSelectedDinner');
            $this->db->from('nas_router');
            $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
            $this->db->group_start();
            $this->db->where('staff.created_by', $user_id);
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
            $this->db->where('MONTH(nas_router.created_at)', date('m'));
            $this->db->where('YEAR(nas_router.created_at)', date('Y'));

            return $this->db->get()->row_array();
        }

        // Regular user
        $this->db->select('SUM(selected_dinner) as TotalSelectedDinner');
        $this->db->from('nas_router');
        $this->db->where('assign_to', $user_id);
        $this->db->where('MONTH(created_at)', date('m'));
        $this->db->where('YEAR(created_at)', date('Y'));

        return $this->db->get()->row_array();
    }

    public function get_total_meals($user_id = null): ?array
    {
        if ($user_id === null) {
            return ['TotalMeals' => 0];
        }

        // Get role
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();

        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            // Superadmin: all users
            $this->db->select('
                SUM(selected_breakfast) + 
                SUM(selected_lunch) + 
                SUM(selected_dinner) AS TotalMeals');
            $this->db->from('nas_router');
            $this->db->where('MONTH(created_at)', date('m'));
            $this->db->where('YEAR(created_at)', date('Y'));
            return $this->db->get()->row_array();
        }

        if (in_array($role, [2, 3])) {
            // Manager or Role 3: own + staff
            $this->db->select('
                SUM(nas_router.selected_breakfast) + 
                SUM(nas_router.selected_lunch) + 
                SUM(nas_router.selected_dinner) AS TotalMeals');
            $this->db->from('nas_router');
            $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
            $this->db->group_start();
            $this->db->where('staff.created_by', $user_id);
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
            $this->db->where('MONTH(nas_router.created_at)', date('m'));
            $this->db->where('YEAR(nas_router.created_at)', date('Y'));
            return $this->db->get()->row_array();
        }

        // Regular user
        $this->db->select('
            SUM(selected_breakfast) + 
            SUM(selected_lunch) + 
            SUM(selected_dinner) AS TotalMeals');
        $this->db->from('nas_router');
        $this->db->where('assign_to', $user_id);
        $this->db->where('MONTH(created_at)', date('m'));
        $this->db->where('YEAR(created_at)', date('Y'));
        return $this->db->get()->row_array();
    }


    public function get_total_meal_cost($user_id = null): ?array
    {
        if ($user_id === null) {
            return ['TotalMealCost' => 0];
        }

        // Get role of the user
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();

        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            // Superadmin: sum all users' meal cost
            $this->db->select('SUM(total_amount) as TotalMealCost');
            $this->db->from('nas_router');
            $this->db->where('MONTH(created_at)', date('m'));
            $this->db->where('YEAR(created_at)', date('Y'));
            return $this->db->get()->row_array();
        }

        if (in_array($role, [2, 3])) {
            // Manager (2) or Role 3: include own + created staff
            $this->db->select('SUM(nas_router.total_amount) as TotalMealCost');
            $this->db->from('nas_router');
            $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
            $this->db->group_start();
            $this->db->where('staff.created_by', $user_id);
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
            $this->db->where('MONTH(nas_router.created_at)', date('m'));
            $this->db->where('YEAR(nas_router.created_at)', date('Y'));
            return $this->db->get()->row_array();
        }

        // Regular user: only their own total cost
        $this->db->select('SUM(total_amount) as TotalMealCost');
        $this->db->from('nas_router');
        $this->db->where('assign_to', $user_id);
        $this->db->where('MONTH(created_at)', date('m'));
        $this->db->where('YEAR(created_at)', date('Y'));
        return $this->db->get()->row_array();
    }

    public function get_income_meal_cost($user_id = null): ?array
    {
        if ($user_id === null) {
            return ['IncomeTotal' => 0];
        }

        // Get role of the user
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();

        $role = $role_data['role'] ?? null;

        if ($role == 1) {
            // Superadmin: sum all users' meal cost
            $this->db->select('SUM(total_amount) as IncomeTotal');
            $this->db->from('nas_router');
            $this->db->where('MONTH(created_at)', date('m'));
            $this->db->where('YEAR(created_at)', date('Y'));
            return $this->db->get()->row_array();
        }

        if (in_array($role, [2, 3])) {
            // Manager (2) or Role 3: include own + created staff
            $this->db->select('SUM(nas_router.total_amount) as IncomeTotal');
            $this->db->from('nas_router');
            $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
            $this->db->group_start();
            $this->db->where('staff.created_by', $user_id);
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
            $this->db->where('MONTH(nas_router.created_at)', date('m'));
            $this->db->where('YEAR(nas_router.created_at)', date('Y'));
            return $this->db->get()->row_array();
        }

        // Regular user: only their own total cost
        $this->db->select('SUM(total_amount) as IncomeTotal');
        $this->db->from('nas_router');
        $this->db->where('assign_to', $user_id);
        $this->db->where('MONTH(created_at)', date('m'));
        $this->db->where('YEAR(created_at)', date('Y'));
        return $this->db->get()->row_array();
    }
    public function get_meal_cost($user_id = null): ?array
    {
        if ($user_id === null) {
            return [
                'BreakfastCost' => 0,
                'LunchCost' => 0,
                'DinnerCost' => 0,
            ];
        }

        // Get user role
        $this->db->select('role');
        $this->db->from('login_credential');
        $this->db->where('user_id', $user_id);
        $role_data = $this->db->get()->row_array();
        $role = $role_data['role'] ?? null;

        // Build query
        $this->db->select([
            'SUM(nas_router.breakfast_cost * nas_router.selected_breakfast) as BreakfastCost',
            'SUM(nas_router.lunch_cost * nas_router.selected_lunch) as LunchCost',
            'SUM(nas_router.dinner_cost * nas_router.selected_dinner) as DinnerCost'
        ]);
        $this->db->from('nas_router');

        if ($role == 1) {
            // Superadmin: no filter
        } elseif (in_array($role, [2, 3])) {
            $this->db->join('staff', 'staff.id = nas_router.assign_to', 'left');
            $this->db->group_start();
            $this->db->where('staff.created_by', $user_id);
            $this->db->or_where('nas_router.assign_to', $user_id);
            $this->db->group_end();
        } else {
            $this->db->where('nas_router.assign_to', $user_id);
        }

        $this->db->where('MONTH(nas_router.created_at)', date('m'));
        $this->db->where('YEAR(nas_router.created_at)', date('Y'));

        return $this->db->get()->row_array();
    }


    public function get_this_month_income($user_id = null): float
    {
        $this->db->select('IFNULL(SUM(credit_in), 0) as total_income')
            ->where('MONTH(datetime)', date('m'))
            ->where('YEAR(datetime)', date('Y'));

        if ($user_id) {

            $this->db->where('manager_id', $user_id);
        }

        $result = $this->db->get('payment_logs')->row_array();
        return $result['total_income'];
    }

    public function get_income_total($user_id = null, $role_id = null)
    {
        $this->db->select('IFNULL(SUM(credit_in), 0) as total_income');
        $this->db->from('payment_logs');
        $this->db->where('MONTH(datetime)', date('m'));
        $this->db->where('YEAR(datetime)', date('Y'));

        // Only filter if not superadmin
        if (!($role_id == 1 && $user_id == 1)) {
            $this->db->where('manager_id', $user_id);
        }

        $result = $this->db->get()->row_array();
        return floatval($result['total_income']);
    }
    public function get_expense_total($user_id = null, $role_id = null)
    {
        $this->db->select('IFNULL(SUM(price), 0) as total_expense');
        $this->db->from('supplier');
        $this->db->where('MONTH(date)', date('m'));
        $this->db->where('YEAR(date)', date('Y'));

        // Only filter if not superadmin
        if (!($role_id == 1 && $user_id == 1)) {
            $this->db->where('prepared_by', $user_id);
        }

        $result = $this->db->get()->row_array();
        return floatval($result['total_expense']);
    }

    public function get_this_month_income_total()

    {
        $query = $this->db->select('IFNULL(SUM(credit_in),0) as total_income')
            ->where('MONTH(datetime)', date('m'))
            ->where('YEAR(datetime)', date('Y'))
            ->where('manager_id', get_loggedin_user_id())
            ->get('payment_logs')->row_array();
        return $query['total_income'];
    }

    public function get_previous_month_income($user_id = null): int
    {
        $this->db->select('IFNULL(COUNT(id), 0) as total_zone');

        if ($user_id) {
            $this->db->where('manager_id', $user_id);
        }

        $result = $this->db->get('zones')->row_array();
        return (int) $result['total_zone'];
    }

    public function get_total_zones($user_id = null): int
    {
        $this->db->select('IFNULL(COUNT(id), 0) as total_zones')
            ->from('zones');

        if (in_array(loggedin_role_id(), [ROLE_RESELLER_ID, ROLE_SUBRESELLER_ID]) && $user_id) {
            $this->db->where('zones.assign_to', $user_id);
        }

        $result = $this->db->get()->row_array();
        return (int) $result['total_zones'];
    }



    public function get_previous_month_total_income($user_id = null): float
    {
        $previous_month = date('m', strtotime('first day of last month'));
        $previous_year = date('Y', strtotime('first day of last month'));

        $query = $this->db->select('IFNULL(SUM(credit_in),0) as total_income')

            ->where('MONTH(datetime)', $previous_month)
            ->where('YEAR(datetime)', $previous_year)
            ->where('manager_id', get_loggedin_user_id())
            ->get('payment_logs')->row_array();
        return $query['total_income'];
    }



    /**
     * @return array|null
     */
    public function get_this_month_bill_total()
    {
        // this month of this year
        $query = $this->db->select('IFNULL(SUM(total-discount+tax_amount),0) as total_bill')
            ->join('staff', 'staff.id=billing.staff_id')
            ->where('MONTH(billing.date)', date('m'))
            ->where('YEAR(billing.date)', date('Y'))
            ->where('staff.assign_to', get_loggedin_user_id())
            ->get('billing')->row_array();
        return $query['total_bill'];
    }
    public function get_this_month_bill($user_id = null): float
    {
        $this->db->select('IFNULL(SUM(total - discount + tax_amount), 0) as total_bill')
            ->join('staff', 'staff.id = billing.staff_id')
            ->where('MONTH(billing.date)', date('m'))
            ->where('YEAR(billing.date)', date('Y'));

        if ($user_id) {
            // Filter bills by assign_to for resellers and sub-resellers
            $this->db->where('staff.assign_to', $user_id);
        }

        $result = $this->db->get('billing')->row_array();
        return $result['total_bill'];
    }



    public function get_previous_month_bill()

    {
        $previous_month = date('m', strtotime('first day of last month'));
        $previous_year = date('Y', strtotime('first day of last month'));

        $query = $this->db->select('IFNULL(SUM(total-discount+tax_amount),0) as total_bill')
            ->join('staff', 'staff.id=billing.staff_id')
            ->where('MONTH(billing.date)', $previous_month)
            ->where('YEAR(billing.date)', $previous_year)
            ->where('staff.assign_to', get_loggedin_user_id())
            ->get('billing')->row_array();
        return $query['total_bill'];
    }

    public function get_previous_month_total_bill($user_id = null): float
    {
        $previous_month = date('m', strtotime('first day of last month'));
        $previous_year = date('Y', strtotime('first day of last month'));

        $this->db->select('IFNULL(SUM(total - discount + tax_amount), 0) as total_bill')
            ->join('staff', 'staff.id = billing.staff_id')
            ->where('MONTH(billing.date)', $previous_month)
            ->where('YEAR(billing.date)', $previous_year);

        if ($user_id) {
            // Filter bills by assign_to for resellers and sub-resellers
            $this->db->where('staff.assign_to', $user_id);
        }

        $result = $this->db->get('billing')->row_array();
        return $result['total_bill'];
    }

    public function get_this_month_total_bill_paid()
    {
        // this month of this year
        $query = $this->db->select('IFNULL(SUM(paid),0) as total_paid')
            ->join('staff', 'staff.id=billing.staff_id')
            ->where('MONTH(billing.date)', date('m'))
            ->where('YEAR(billing.date)', date('Y'))
            ->where('staff.assign_to', get_loggedin_user_id())
            ->get('billing')->row_array();
        return $query['total_paid'];
    }

    public function get_this_month_bill_paid($user_id = null): float
    {
        $this->db->select('IFNULL(SUM(paid), 0) as total_paid')
            ->join('staff', 'staff.id = billing.staff_id')
            ->where('MONTH(billing.date)', date('m'))
            ->where('YEAR(billing.date)', date('Y'));

        if ($user_id) {
            // Filter by assign_to for resellers and sub-resellers
            $this->db->where('staff.assign_to', $user_id);
        }

        $result = $this->db->get('billing')->row_array();
        return $result['total_paid'];
    }



    public function get_this_month_total_bill_due()
    {
        // this month of this year
        $query = $this->db->select('IFNULL(SUM(due),0) as total_due')
            ->join('staff', 'staff.id=billing.staff_id')
            ->where('MONTH(billing.date)', date('m'))
            ->where('YEAR(billing.date)', date('Y'))
            ->where('staff.assign_to', get_loggedin_user_id())
            ->get('billing')->row_array();
        return $query['total_due'];
    }

    public function get_this_month_bill_due($user_id = null): float
    {
        $this->db->select('IFNULL(SUM(total - paid), 0) as total_due')
            ->join('staff', 'staff.id = billing.staff_id')
            ->where('MONTH(billing.date)', date('m'))
            ->where('YEAR(billing.date)', date('Y'));

        if ($user_id) {
            // Filter by assign_to for Resellers and Sub-Resellers
            $this->db->where('staff.assign_to', $user_id);
        }

        $result = $this->db->get('billing')->row_array();
        return $result['total_due'];
    }


    public function get_previous_month_bill_paid()
    {
        // Previous month of the current year
        $previous_month = date('m', strtotime('first day of last month'));
        $previous_year = date('Y', strtotime('first day of last month'));

        $query = $this->db->select('IFNULL(SUM(paid), 0) as total_paid')
            ->join('staff', 'staff.id = billing.staff_id')
            ->where('MONTH(billing.date)', $previous_month)
            ->where('YEAR(billing.date)', $previous_year)
            ->where('staff.assign_to', get_loggedin_user_id())
            ->get('billing')
            ->row_array();
        return $query['total_paid'];
    }

    public function get_previous_month_total_bill_paid($user_id = null): float
    { // Previous month of the current year
        $previous_month = date('m', strtotime('first day of last month'));
        $previous_year = date('Y', strtotime('first day of last month'));
        $this->db->select('IFNULL(SUM(paid), 0) as total_paid')
            ->join('staff', 'staff.id = billing.staff_id')
            ->where('MONTH(billing.date)', $previous_month)
            ->where('YEAR(billing.date)', $previous_year);
        if ($user_id) {
            // Filter bills by assign_to for resellers and sub-resellers
            $this->db->where('staff.assign_to', $user_id);
        }
        $result = $this->db->get('billing')->row_array();
        return $result['total_paid'];
    }


    public function get_previous_month_bill_due($user_id = null): float
    {
        // this month of this year
        $previous_month = date('m', strtotime('first day of last month'));
        $previous_year = date('Y', strtotime('first day of last month'));

        $query = $this->db->select('IFNULL(SUM(due),0) as total_due')
            ->join('staff', 'staff.id=billing.staff_id')
            ->where('MONTH(billing.date)', $previous_month)
            ->where('YEAR(billing.date)', $previous_year)
            ->where('staff.assign_to', get_loggedin_user_id())
            ->get('billing')->row_array();
        return $query['total_due'];
    }

    public function get_previous_month_total_bill_due($user_id = null): float
    {
        $previous_month = date('m', strtotime('first day of last month'));
        $previous_year = date('Y', strtotime('first day of last month'));
        $this->db->select('IFNULL(SUM(due),0) as total_due')
            ->join('staff', 'staff.id=billing.staff_id')
            ->where('MONTH(billing.date)', $previous_month)
            ->where('YEAR(billing.date)', $previous_year);
        if ($user_id) {
            // Filter bills by assign_to for resellers and sub-resellers
            $this->db->where('staff.assign_to', $user_id);
        }

        $result = $this->db->get('billing')->row_array();
        return $result['total_due'];
    }



    public function get_previous_month_total_expense($user_id = null)
    {
        $this->db->select('IFNULL(SUM(credit_out), 0) as total_expense')
            ->where('MONTH(datetime)', date('m', strtotime('-1 month')))
            ->where('YEAR(datetime)', date('Y'));

        if ($user_id) {
            $this->db->where('manager_id', $user_id);
        }

        $result = $this->db->get('payment_logs')->row_array();
        return $result['total_expense'];
    }
    public function get_single_employee($id = null)
    {
        $this->db->select('staff.*,staff_designation.name as designation_name,staff_department.name as department_name,login_credential.role as role_id,login_credential.active,login_credential.username, roles.name as role, staff.assigned_admin');
        $this->db->select('packages.name as package_name');
        $this->db->select('nas_router.name as nas_name');
        $this->db->from('staff');
        $this->db->join('login_credential', 'login_credential.user_id = staff.id and login_credential.role != "7"', 'inner');
        $this->db->join('roles', 'roles.id = login_credential.role', 'left');
        $this->db->join('staff_designation', 'staff_designation.id = staff.designation', 'left');
        $this->db->join('staff_department', 'staff_department.id = staff.department', 'left');
        $this->db->join('packages', 'packages.id = staff.package_id', 'left');
        $this->db->join('nas_router', 'nas_router.id = staff.nas_id', 'left');
        $this->db->where('staff.id', $id);
        if (in_array(loggedin_role_id(), [ROLE_RESELLER_ID, ROLE_SUBRESELLER_ID])) {
            $this->db->where('staff.assign_to', get_loggedin_user_id());
        }
        $query = $this->db->get();
        return $query->row_array();
    }
}
