<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Class Base_Model
 */
class Base_Model extends CI_Model
{
    /**
     * @var string Table name
     */
    protected $table;

    /**
     * @var string Primary key
     */
    protected $primaryKey = 'id';

    /**
     * @var string Return type
     */
    protected $returnType = 'array';

    /**
     * @var array
     */
    protected $allowedFields = [];

    /**
     * @var bool Use timestamps
     */
    protected $useTimestamps = false;
    protected $created_at = 'created_at';
    protected $updated_at = 'updated_at';
    protected $deleted_at = 'deleted_at';

    /**
     * @var bool Use soft delete
     */
    protected $useSoftDelete = false;

    /**
     * @var string
     */
    private $baseSelect = '*';

    /**
     * @var true
     */
    private $onlyTrashed;

    public function __construct()
    {
        parent::__construct();

        // if the table name is not set, use class name without model part
        if (!$this->table) {
            $this->table = strtolower(str_replace('_model', 's', get_class($this)));
        }
    }

    /**
     * where
     */
    public function where($key, $value = NULL, $escape = NULL): Base_Model
    {
        $this->db->where($key, $value, $escape);
        return $this;
    }

    /**
     * @param int|null $id
     * @return mixed
     */
    public function find(int $id = null)
    {
        if ($id) {
            $this->db->where('id', $id);
        }

        if ($this->useTimestamps and $this->useSoftDelete) {
            $this->db->where($this->deleted_at, null);
        }

        $result = $this->db->get($this->table);


        if ($this->returnType === 'object') {
            return $result->row();
        }

        return $result->row_array();
    }

    /**
     * @return mixed
     */
    public function findAll()
    {
        $this->db->select($this->baseSelect);

        /**
         * If useTimestamps and useSoftDelete is true
         *      If onlyTrashed is true
         *          show only deleted items
         *      else
         *          don't show deleted items
         */
        if ($this->useTimestamps && $this->useSoftDelete) {
            if ($this->onlyTrashed) {
                $this->db->where($this->deleted_at . ' IS NOT NULL'); // show only deleted items
            } else {
                $this->db->where($this->deleted_at, null); // don't show deleted items
            }
        }

        $result = $this->db->get($this->table);

        if ($this->returnType === 'object') {
            return $result->result();
        }

        return $result->result_array();
    }

    /**
     * @param $data
     * @return false
     */
    public function insert($data): bool
    {
        if (empty($data)) {
            return false;
        }

        // only allowed fields can be inserted into a database other data fields will be ignored
        if (!empty($this->allowedFields)) {
            $data = array_intersect_key($data, array_flip($this->allowedFields));
        }

        if ($this->useTimestamps) {
            if ($this->created_at)
                $data[$this->created_at] = date('Y-m-d H:i:s');
        }

        $this->db->insert($this->table, $data);
        return $this->db->insert_id();
    }

    /**
     * @param int $id
     * @param $data
     * @return bool
     */
    public function update(int $id, $data): bool
    {
        if (empty($data)) {
            throw new InvalidArgumentException('Data cannot be empty');
        }

        // only allowed fields can be inserted into a database other data fields will be ignored
        if (!empty($this->allowedFields)) {
            $data = array_intersect_key($data, array_flip($this->allowedFields));
        }

        if ($this->useTimestamps) {
            $data[$this->updated_at] = date('Y-m-d H:i:s');
        }

        $this->db->where('id', $id);
        return $this->db->update($this->table, $data);
    }

    public function delete(int $id)
    {
        $this->db->where('id', $id);

        if ($this->useTimestamps and $this->useSoftDelete) {
            $data[$this->deleted_at] = date('Y-m-d H:i:s');
            return $this->db->update($this->table, $data);
        }

        return $this->db->delete($this->table);
    }

    public function baseSelect(string $select): Base_Model
    {
        $this->baseSelect = $select;
        return $this;
    }

    public function limit(int $limit, int $offset = 0): Base_Model
    {
        $this->db->limit($limit, $offset);
        return $this;
    }

    public function withTrashed(): Base_Model
    {
        $this->useSoftDelete = false;
        return $this;
    }

    public function onlyTrashed(): Base_Model
    {
        $this->onlyTrashed = true;
        return $this;
    }

    public function orderBy(string $orderBy, string $order = 'ASC'): Base_Model
    {
        $this->db->order_by($orderBy, $order);
        return $this;
    }

    public function whereGroupStart(): Base_Model
    {
        $this->db->group_start();
        return $this;
    }

    public function orWhere(string $key, $value): Base_Model
    {
        $this->db->or_where($key, $value);
        return $this;
    }

    public function whereGroupEnd(): Base_Model
    {
        $this->db->group_end();
        return $this;
    }
}

class MY_Model extends CI_Model
{

    function __construct()
    {
        parent::__construct();
    }

    function get_list($table_name, $where_array = NULL, $single = FALSE, $columns = NULL)
    {
        if ($columns)
            $this->db->select($columns);

        if (!empty($where_array))
            $this->db->where($where_array);

        if ($single) {
            $method = 'row_array';
        } else {
            $method = 'result_array';
            $this->db->order_by('id', 'ASC');
        }
        return $this->db->get($table_name)->$method();
    }
}
