<?php defined('BASEPATH') or exit('No direct script access allowed');

class Package extends Admin_Controller
{
    private $main_menu = 'configuration';
    private $permission = 'package';


    private $packageTypeList = [
        'low',
        'medium',
        'high'
    ];

    public function __construct()
    {
        parent::__construct();

        $this->load->model('package_model');
        $this->load->model('zone_model');
        $this->load->model('employee_model');
    }

    /**
     * Show all packages
     * Create new package
     * Store new package
     *
     * @return object|string
     */
    public function index()
    {
        // Check permission to view packages
        if (!get_permission($this->permission, 'is_view')) {
            access_denied();
        }

        $userID = get_loggedin_user_id();

        // Get logged-in user's name and ID from staff table
        $this->db->select('id, name');
        $this->db->from('staff');
        $this->db->where('id', $userID);
        $loggedUser = $this->db->get()->row();

        if ($this->input->method(true) === 'POST' && get_permission($this->permission, 'is_add')) {
            if ($this->packageStoreValidation()) {
                $postData = $this->input->post();

                // For admin (userID == 1), allow selected manager_id from form
                // For others, force manager_id to logged-in user
                if ($userID != 1) {
                    $postData['manager_id'] = $userID;
                }

                // Handle product selection (array to comma-separated string)
                $selectedProducts = $this->input->post('product');
                $postData['product'] = !empty($selectedProducts) ? implode(',', $selectedProducts) : null;

                $this->package_model->insert($postData);

                set_alert('success', translate('package_added_successfully'));
                redirect('package');
            }

            set_alert('error', translate('please_fix_form_data_and_try_again'));
        }

        $this->data['title'] = translate('Meal Package');
        $this->data['main_menu'] = $this->main_menu;
        $this->data['sub_page'] = 'package/index';
        $this->data['productlist'] = $this->package_model->get_product_list();

        // Pass filtered packages to view (admin sees all, others only their own)
        $this->data['packages'] = $this->package_model->findAll_filtered(null, $userID);

        $this->data['packageTypeList'] = $this->packageTypeList;

        // Pass logged user info for view
        $this->data['manager_id'] = $loggedUser->id ?? '';
        $this->data['manager_name'] = $loggedUser->name ?? '';

        if ($userID == 1) {
            $this->db->select('staff.id, staff.name');
            $this->db->from('staff');
            $this->db->join('login_credential', 'login_credential.user_id = staff.id', 'inner');
            $this->db->where('login_credential.role', 2);
            $this->data['employees'] = $this->db->get()->result();
        } else {
            // For other users, just show logged-in user
            $this->db->select('id, name');
            $this->db->from('staff');
            $this->db->where('id', $userID);
            $this->data['employees'] = $this->db->get()->result();
        }

        return $this->load->view('layout/index', $this->data);
    }


    /**
     * edit package ui
     * update package
     * @return object|string
     */
    public function edit(int $id)
    {
        // check access permission
        if (!get_permission($this->permission, 'is_edit')) {
            access_denied();
        }
        $userID = get_loggedin_user_id();

        // Get logged-in user's name and ID from staff table
        $this->db->select('id, name');
        $this->db->from('staff');
        $this->db->where('id', $userID);
        $loggedUser = $this->db->get()->row();
        
        $package = $this->package_model->whereManagerId(get_loggedin_user_id())->find($id);
        if (is_null($package)) {
            redirect('package');
        }

        // Process Post Request
        if ($this->input->method(true) === 'POST') {
            if ($this->packageStoreValidation()) {

                $postData = $this->input->post();

                // Convert product array to comma-separated string if present
                if (!empty($postData['product']) && is_array($postData['product'])) {
                    $postData['product'] = implode(',', $postData['product']);
                }

                $this->package_model->update($id, $postData);
                set_alert('success', translate('package_updated_successfully'));
                redirect('package');
            }

            set_alert('error', translate('please_fix_form_data_and_try_again'));
        }
        // Pass logged user info for view
        $this->data['manager_id'] = $loggedUser->id ?? '';
        $this->data['manager_name'] = $loggedUser->name ?? '';

        // Prepare product array for form pre-fill (split from comma string)

        // Process Get Request
        $this->data['title'] = translate('Meal Package');
        $this->data['action'] = 'edit';
        $this->data['main_menu'] = $this->main_menu;
        $this->data['sub_page'] = 'package/index';
        $this->data['productlist'] = $this->package_model->get_product_list();
        $this->data['package'] = $package;
        $this->data['employees'] = $this->employee_model->get_admin_reseller_subreseller_list(ROLE_ADMIN_ID, get_loggedin_user_id());
        $this->data['packageTypeList'] = $this->packageTypeList;

        return $this->load->view('layout/index', $this->data);
    }


    /**
     * @param int $nasId
     * @return null
     */
public function check_staff_status()
{
    $id = $this->input->post('id');

    $this->load->model('package_model'); // replace with your actual model
    $staff = $this->package_model->get($id); // adjust method if needed

    if ($staff && strtolower($staff['status']) === 'active') {
        echo json_encode(['status' => 'active']);
    } else {
        echo json_encode(['status' => 'inactive']);
    }
}


    public function ajax_packages_by_reseller_subreseller_id(int $reseller_subreseller_id)
    {
        $RoleID = loggedin_role_id();
        // check if ajax request
        /*  if (!$this->input->is_ajax_request()) {
             return show_error(translate('No direct script access allowed')); // Prevent direct access
         }
  */
        // check access permission
        if (!get_permission('package', 'is_view')) {
            $response = [
                'status' => 'error',
                'message' => translate('access_denied'),
            ];
            return $this->output->set_status_header(403)->set_content_type('application/json')->set_output(json_encode($response));
        }

        // check if packages exists
        $packages = $this->package_model->get_package_list_individual($RoleID, $reseller_subreseller_id);

        if (empty($packages)) {
            $response = [
                'status' => 'error',
                'message' => translate('packages_not_found'),
            ];
            return $this->output->set_status_header(404)->set_content_type('application/json')->set_output(json_encode($response));
        } else {

            $zones = $this->zone_model->get_zone_list_individual($RoleID, $reseller_subreseller_id);
            // Process the data...
            $response = array(
                'status' => 'success',
                'packages' => $packages,
                'zones' => $zones,
                // Add more data to the response as needed...
            );

            return $this->output->set_content_type('application/json')->set_output(json_encode($response));
        }
    }

    private function packageStoreValidation()
    {
        $this->form_validation->set_rules([
            [
                'field' => 'name',
                'label' => translate('package_name'),
                'rules' => 'trim|required|min_length[3]|max_length[255]',
            ],
            [
                'field' => 'type',
                'label' => translate('package_type'),
                'rules' => 'trim|required|in_list[low,medium,high]',
            ],
            [
                'field' => 'manager_id',
                'label' => translate('manager'),
                'rules' => 'trim|required|integer',
            ],

            [
                'field' => 'sale_price',
                'label' => translate('sale_price'),
                'rules' => 'trim|required',
            ],
        ]);

        return $this->form_validation->run();
    }
}
