<?php
defined('BASEPATH') or exit('No direct script access allowed');


class Nas extends Admin_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('nas_model');
        $this->load->model('employee_model');
        $this->load->model('zone_model');
        $this->load->model('email_model');
        $this->load->model('inventory_model');
        $this->load->model('package_model');
        $this->load->model('smsmanager_model');
        $this->load->model('olt_model');
        $this->load->library('routerosapi');
    }

    public function index()
    {
        if (!get_permission('nas', 'is_view')) {
            access_denied();
        }
        $logged_id = get_loggedin_user_id();

        $userID = get_loggedin_user_id();
        $RoleID = loggedin_role_id();


        $search_user_id = $this->input->post('client_id');
        $search_user_id = empty($search_user_id) ? $logged_id : $search_user_id;

        $daterange = explode(' - ', $this->input->post('daterange'));
        if (isset($daterange) && !empty($daterange)) {
            $start = isset($daterange[0]) ? date("Y-m-d 00:00:00", strtotime($daterange[0])) : date("Y-m-d 00:00:00");
            $end = isset($daterange[1]) ? date("Y-m-d 23:59:59", strtotime($daterange[1])) : date("Y-m-d 23:59:59");
        } else {
            $start = date("Y-m-d 00:00:00");
            $end = date("Y-m-d 23:59:59");
        }

        $this->load->library('datatables');
        $dt_authors = $this->datatables->init();
        $dt_authors->get_meal_order_logs_filter($search_user_id, $start, $end, false);

        $dt_authors
            ->style(array(
                'class' => 'table table-bordered table-striped dataTable dtr-inline table-hover table-condensed dataTable no-footer',
            ))
            ->set_options('responsive', 'true')
            ->set_options('autoWidth', 'false')
            ->set_options('pagingType', '\'full_numbers\'')
            ->set_options('pageLength', '10')
            ->set_options('order', '[[0, \'desc\']]')
            ->set_options('lengthMenu', '[[ 10, 25, 50, 1000, 5000 ],[ \'10\', \'25\', \'50\', \'1000\', \'5000\' ]]')
            ->set_options('dom', '\'<"container-fluid"<"row"<"col-sm-4"l><"col-sm-4"B><"col-sm-4"f>>>rtip\'')
            ->set_options('buttons', '["copy", "csv", "excel", "pdf", "print", "colvis"]')
            ->set_options('ajax.data', 'd.client_id = $(\'#client_id\').val();')
            ->set_options('ajax.data', 'd.daterange = $(\'#daterange\').val();')
            ->column('User ID', 'id')
            ->column('Client Name', 'student_name')
            ->column('Client ID', 'idd')
            ->column('Branch Name', 'manager_name')
            ->column('Branch ID', 'branch_merchant_no')
            ->column('Package', 'package_name')
            ->column('Breakfast', 'selected_breakfast')
            ->column('Lunch', 'selected_lunch')
            ->column('Dinner', 'selected_dinner')
            ->column('Meal Cost', 'total_amount')
            ->column('Order Dates', 'selected_dates')
            ->column('Refund', 'id', function ($data, $row) {
                $buttons = '';
                $selectedDate = DateTime::createFromFormat('Y-m-d', $row['selected_dates']);
                $today = new DateTime('today');

                if (!$selectedDate) {
                    return '<span class="badge badge-warning">Invalid Date</span>';
                }

                $diffObj = $selectedDate->diff($today);

                if ($diffObj->invert === 1 && $diffObj->days >= 2) {
                    $hasBreakfast = !empty($row['selected_breakfast']);
                    $hasLunch = !empty($row['selected_lunch']);
                    $hasDinner = !empty($row['selected_dinner']);

                    if (!$hasBreakfast && !$hasLunch && !$hasDinner) {
                        return '<span class="badge badge-secondary">All Meals Refunded</span>';
                    }

                    // Show refund button if any meal is still available
                    $buttons .= '<a href="javascript:void(0);" class="btn btn-primary btn-circle icon" data-toggle="tooltip" title="Request Refund" onclick="make_refund_request(' . $row['id'] . ')"><i class="fas fa-undo"></i></a>';
                } else {
                    $buttons .= '<span class="badge badge-danger">Request Timeout</span>';
                }

                return $buttons;
            });


        $this->datatables->create('dt_authors', $dt_authors);

        $this->data['title'] = translate('Meal Booking');
        $this->data['package_list'] = $this->package_model->get_package_list_individual($RoleID, $userID);
        $this->data['zones_list'] = $this->zone_model->get_zone_list_individual($RoleID, $userID);
        $this->data['reseller_subreseller_list'] = $this->employee_model->get_reseller_subreseller_list($RoleID, $userID, '');
        $this->data['students'] = $this->nas_model->get_students_by_manager($userID);
        $this->data['client_id_list'] = $this->employee_model->get_only_client_list('', $logged_id, '');
        $this->data['studentlist'] = $this->nas_model->get_all_students();

        $this->data['sub_page'] = 'nas/index';
        $this->data['main_menu'] = 'nas';
        $this->load->view('layout/index', $this->data);
    }




    public function refund_list()
    {
        if (!get_permission('nas_refund', 'is_view')) {
            access_denied();
        }
        $logged_id = get_loggedin_user_id();

        $userID = get_loggedin_user_id();
        $RoleID = loggedin_role_id();


        $search_user_id = $this->input->post('client_id');
        $search_user_id = empty($search_user_id) ? $logged_id : $search_user_id;

        $daterange = explode(' - ', $this->input->post('daterange'));
        if (isset($daterange) && !empty($daterange)) {
            $start = isset($daterange[0]) ? date("Y-m-d 00:00:00", strtotime($daterange[0])) : date("Y-m-d 00:00:00");
            $end = isset($daterange[1]) ? date("Y-m-d 23:59:59", strtotime($daterange[1])) : date("Y-m-d 23:59:59");
        } else {
            $start = date("Y-m-d 00:00:00");
            $end = date("Y-m-d 23:59:59");
        }

        $this->load->library('datatables');
        $dt_authors = $this->datatables->init();
        $dt_authors->get_client_refund_logs_filter($search_user_id, $start, $end, false);

        $dt_authors
            ->style(array(
                'class' => 'table table-bordered table-striped dataTable dtr-inline table-hover table-condensed dataTable no-footer',
            ))
            ->set_options('responsive', 'true')
            ->set_options('autoWidth', 'false')
            ->set_options('pagingType', '\'full_numbers\'')
            ->set_options('pageLength', '10')
            ->set_options('order', '[[0, \'desc\']]')
            ->set_options('lengthMenu', '[[ 10, 25, 50, 1000, 5000 ],[ \'10\', \'25\', \'50\', \'1000\', \'5000\' ]]')
            ->set_options('dom', '\'<"container-fluid"<"row"<"col-sm-4"l><"col-sm-4"B><"col-sm-4"f>>>rtip\'')
            ->set_options('buttons', '["copy", "csv", "excel", "pdf", "print", "colvis"]')
            ->set_options('ajax.data', 'd.client_id = $(\'#client_id\').val();')
            ->set_options('ajax.data', 'd.daterange = $(\'#daterange\').val();')
            ->column('User ID', 'id')
            ->column('Client Name', 'student_name')
            ->column('Client ID', 'idd')
            ->column('Branch Name', 'manager_name')
            ->column('Branch ID', 'manager_id')
            ->column('Package', 'package_name')
            ->column('Breakfast', 'selected_breakfast')
            ->column('Lunch', 'selected_lunch')
            ->column('Dinner', 'selected_dinner')
            ->column('Meal Cost', 'total_amount')
            ->column('Order Dates', 'selected_dates')
            ->column('Refund Time', 'refunded_at')
            ->column('Status', 'status', function ($value, $row) {
                if ($value == 1) {
                    return '<span class="badge" style=" background-color: #d4edda; color: #155724;padding: 0.4em 0.75em; border-radius: 0.35rem; font-weight: 600;display: inline-flex;
            align-items: center;
            gap: 0.3em;
        ">
            <i class="fas fa-check-circle"></i> Approved
        </span>';
                } elseif ($value == 2) {
                    return '<span class="badge" style="
            background-color: #f8d7da; 
            color: #721c24; 
            padding: 0.4em 0.75em; 
            border-radius: 0.35rem; 
            font-weight: 600;
            display: inline-flex;
            align-items: center;
            gap: 0.3em;
        ">
            <i class="fas fa-times-circle"></i> Cancelled
        </span>';
                } else {
                    return '<span class="badge" style="
            background-color: #fff3cd; 
            color: #856404; 
            padding: 0.4em 0.75em; 
            border-radius: 0.35rem; 
            font-weight: 600;
            display: inline-flex;
            align-items: center;
            gap: 0.3em;
        ">
            <i class="fas fa-clock"></i> Pending
        </span>';
                }
            })
            ->column('Action', 'id', function ($data, $row) {
                $buttons = '';

                if (get_permission('refund_client', 'is_edit')) {
                    // Hide refund button if already approved
                    if ($row['status'] != 1 && $row['status'] != 2) {
                        $selectedDate = DateTime::createFromFormat('Y-m-d', $row['selected_dates']);
                        $today = new DateTime('today');

                        if ($selectedDate && $today < $selectedDate && $selectedDate->diff($today)->days >= 1) {
                            $buttons .= '<a href="javascript:void(0);" class="btn btn-primary btn-circle icon" data-toggle="tooltip" data-original-title="Refund" onclick="make_refund(' . $row['id'] . ')"><i class="fas fa-undo"></i></a> ';
                        }
                    }

                    // Hide cancel button if already cancelled
                    if ($row['status'] != 1 && $row['status'] != 2) {
                        $buttons .= '<a href="javascript:void(0);" class="btn btn-danger btn-circle icon" data-toggle="tooltip" data-original-title="Cancel" onclick="cancel_refund(' . $row['id'] . ')"><i class="fas fa-times"></i></a>';
                    }
                }

                return $buttons;
            });


        $this->datatables->create('dt_authors', $dt_authors);



        $this->load->model('nas_model'); // Ensure model is loaded
        $this->data['refunds'] = $this->nas_model->get_refund_list($RoleID, $userID);
        $this->data['client_id_list'] = $this->employee_model->get_only_client_list('', $logged_id, '');
        $this->data['students'] = $this->nas_model->get_students_by_manager($userID);
        $this->data['studentlist'] = $this->nas_model->get_all_students();


        $this->data['title'] = translate('Refund List');
        $this->data['sub_page'] = 'nas/refund_list';
        $this->data['main_menu'] = 'nas';


        $this->load->view('layout/index', $this->data);
    }
    public function request_refund($id = '')
    {
        $record = $this->db->get_where('nas_router', ['id' => $id])->row_array();
        if (!$record) {
            echo json_encode(['status' => 'error', 'message' => 'Record not found.']);
            return;
        }

        $selected_meals = $this->input->post('meals');
        if (empty($selected_meals)) {
            echo json_encode(['status' => 'error', 'message' => 'No meals selected.']);
            return;
        }

        $existingRefunds = $this->db->get_where('refund', [
            'request_id' => $id,
            'selected_dates' => $record['selected_dates']
        ])->result_array();

        $alreadyRefunded = [
            'breakfast' => false,
            'lunch' => false,
            'dinner' => false
        ];

        foreach ($existingRefunds as $row) {
            if (!empty($row['selected_breakfast'])) $alreadyRefunded['breakfast'] = true;
            if (!empty($row['selected_lunch']))     $alreadyRefunded['lunch'] = true;
            if (!empty($row['selected_dinner']))    $alreadyRefunded['dinner'] = true;
        }


        $mealsToRefund = array_filter($selected_meals, function ($meal) use ($alreadyRefunded) {
            return !$alreadyRefunded[$meal];
        });

        if (empty($mealsToRefund)) {
            echo json_encode(['status' => 'error', 'message' => 'Selected meals already requested for refund.']);
            return;
        }


        $refund_amount = 0;
        $logged_user_id = get_loggedin_user_id();

        $staff = $this->db->get_where('staff', ['id' => $logged_user_id])->row_array();
        $assign_to_id = $staff['assign_to'] ?? null;

        $manager = $this->db->get_where('staff', ['id' => $assign_to_id])->row_array();
        $manager_merchant_no = $manager['merchant_no'] ?? '';

        $refund_data = [
            'request_id' => $record['id'],
            'student_name' => $record['student_name'],
            'idd' => $record['idd'],
            'zone_name' => $record['zone_name'] ?? '',
            'manager_name' => $record['manager_name'],
            'package_name' => $record['package_name'],
            'selected_dates' => $record['selected_dates'],
            'refunded_by' => $logged_user_id,
            'refunded_at' => date('Y-m-d H:i:s'),
            'status' => 0,
            'manager_id' => $manager_merchant_no
        ];


        if (in_array('breakfast', $mealsToRefund) && !empty($record['selected_breakfast'])) {
            $refund_data['selected_breakfast'] = $record['selected_breakfast'];
            $refund_amount += $record['breakfast_cost'];
        } else {
            $refund_data['selected_breakfast'] = '';
        }

        if (in_array('lunch', $mealsToRefund) && !empty($record['selected_lunch'])) {
            $refund_data['selected_lunch'] = $record['selected_lunch'];
            $refund_amount += $record['lunch_cost'];
        } else {
            $refund_data['selected_lunch'] = '';
        }

        if (in_array('dinner', $mealsToRefund) && !empty($record['selected_dinner'])) {
            $refund_data['selected_dinner'] = $record['selected_dinner'];
            $refund_amount += $record['dinner_cost'];
        } else {
            $refund_data['selected_dinner'] = '';
        }

        if ($refund_amount <= 0) {
            echo json_encode(['status' => 'error', 'message' => 'No valid meals found for refund.']);
            return;
        }

        $refund_data['total_amount'] = $refund_amount;

        $this->db->insert('refund', $refund_data);

        echo json_encode(['status' => 'success', 'message' => 'Refund request submitted successfully.']);
    }

    public function get_meal_options($id = '')
    {

        $record = $this->db->get_where('nas_router', ['id' => $id])->row_array();
        if (!$record) {
            echo json_encode(['status' => 'error', 'message' => 'Record not found.']);
            return;
        }

        $refunds = $this->db->get_where('refund', [
            'request_id' => $id,
            'selected_dates' => $record['selected_dates']
        ])->result_array();

        $refunded = [
            'breakfast' => false,
            'lunch'     => false,
            'dinner'    => false
        ];

        foreach ($refunds as $r) {
            if (!empty($r['selected_breakfast'])) $refunded['breakfast'] = true;
            if (!empty($r['selected_lunch']))     $refunded['lunch'] = true;
            if (!empty($r['selected_dinner']))    $refunded['dinner'] = true;
        }

        // Return only non-refunded/requested meals
        $meals = [
            'breakfast'      => !empty($record['selected_breakfast']) && !$refunded['breakfast'],
            'lunch'          => !empty($record['selected_lunch']) && !$refunded['lunch'],
            'dinner'         => !empty($record['selected_dinner']) && !$refunded['dinner'],
            'breakfast_cost' => $record['breakfast_cost'],
            'lunch_cost'     => $record['lunch_cost'],
            'dinner_cost'    => $record['dinner_cost'],
        ];

        echo json_encode(['status' => 'success', 'meals' => $meals]);
    }

    public function get_meal($id = '')
    {

        $record = $this->db->get_where('refund', ['id' => $id])->row_array();

        if (!$record) {
            echo json_encode(['status' => 'error', 'message' => 'Refund record not found.']);
            return;
        }

        $meals = [
            'breakfast' => !empty($record['selected_breakfast']),
            'lunch'     => !empty($record['selected_lunch']),
            'dinner'    => !empty($record['selected_dinner']),
        ];

        echo json_encode([
            'status'    => 'success',
            'meals'     => $meals,
            'refund_id' => $record['id'],
        ]);
    }

    public function submit_refund_request()
    {
        $id = $this->input->post('nas_id');
        $selected_meals = $this->input->post('meals');

        if (!$id || empty($selected_meals)) {
            echo json_encode(['status' => 'error', 'message' => 'Please select at least one meal.']);
            return;
        }

        $record = $this->db->get_where('nas_router', ['id' => $id])->row_array();
        if (!$record) {
            echo json_encode(['status' => 'error', 'message' => 'Record not found.']);
            return;
        }

        $data = [
            'id' => $record['id'],
            'student_name' => $record['student_name'],
            'idd' => $record['idd'],
            'zone_name' => $record['zone_name'],
            'manager_name' => $record['manager_name'],
            'package_name' => $record['package_name'],
            'selected_breakfast' => in_array('breakfast', $selected_meals) ? $record['selected_breakfast'] : '',
            'selected_lunch' => in_array('lunch', $selected_meals) ? $record['selected_lunch'] : '',
            'selected_dinner' => in_array('dinner', $selected_meals) ? $record['selected_dinner'] : '',
            'selected_dates' => $record['selected_dates'],
            'total_amount' => 0,
            'refunded_by' => get_loggedin_user_id(),
            'refunded_at' => date('Y-m-d H:i:s'),
            'status' => 0, // Pending
        ];

        $this->db->insert('refund', $data);

        echo json_encode(['status' => 'success', 'message' => 'Refund request submitted.']);
    }
    public function refund_client($id = '')
    {
        if (!get_permission('refund_client', 'is_edit')) {
            access_denied();
        }

        // Fetch refund record by its ID
        $refund = $this->db->get_where('refund', ['id' => $id, 'status' => 0])->row_array();
        if (!$refund) {
            echo json_encode(['status' => 'error', 'message' => 'No pending refund request found.']);
            return;
        }

        // Now fetch the related nas_router record using refund.request_id
        $nasRecord = $this->db->get_where('nas_router', ['id' => $refund['request_id']])->row_array();
        if (!$nasRecord) {
            echo json_encode(['status' => 'error', 'message' => 'Original record not found.']);
            return;
        }

        // Get selected meals
        $selected_meals = $this->input->post('meals');
        if (empty($selected_meals)) {
            echo json_encode(['status' => 'error', 'message' => 'No meals selected.']);
            return;
        }

        // Check already approved meals for same request_id and date
        $approved_rows = $this->db->get_where('refund', [
            'request_id' => $nasRecord['id'],
            'selected_dates' => $refund['selected_dates'],
            'status' => 1
        ])->result_array();

        $already_refunded = [];
        foreach ($approved_rows as $row) {
            if (!empty($row['selected_breakfast'])) $already_refunded[] = 'breakfast';
            if (!empty($row['selected_lunch']))     $already_refunded[] = 'lunch';
            if (!empty($row['selected_dinner']))    $already_refunded[] = 'dinner';
        }

        // Filter out already refunded
        $selected_meals = array_diff($selected_meals, $already_refunded);
        if (empty($selected_meals)) {
            echo json_encode(['status' => 'error', 'message' => 'These meals were already refunded.']);
            return;
        }

        $refund_amount = 0;
        $update_nas_router = [];

        if (in_array('breakfast', $selected_meals) && !empty($refund['selected_breakfast'])) {
            $refund_amount += $nasRecord['breakfast_cost'];
            $update_nas_router['selected_breakfast'] = '';
        }

        if (in_array('lunch', $selected_meals) && !empty($refund['selected_lunch'])) {
            $refund_amount += $nasRecord['lunch_cost'];
            $update_nas_router['selected_lunch'] = '';
        }

        if (in_array('dinner', $selected_meals) && !empty($refund['selected_dinner'])) {
            $refund_amount += $nasRecord['dinner_cost'];
            $update_nas_router['selected_dinner'] = '';
        }

        if ($refund_amount <= 0) {
            echo json_encode(['status' => 'error', 'message' => 'No valid meals to approve.']);
            return;
        }

        // Update refund status to approved
        $this->db->where('id', $id)->update('refund', [
            'status' => 1
        ]);

        // Add refund to user balance
        $this->db->set('user_balance', 'user_balance + ' . $refund_amount, false)
            ->where('id', $nasRecord['assign_to'])
            ->update('staff');

        // Handle NAS router update or delete
        $remaining_breakfast = isset($update_nas_router['selected_breakfast']) ? '' : $nasRecord['selected_breakfast'];
        $remaining_lunch     = isset($update_nas_router['selected_lunch'])     ? '' : $nasRecord['selected_lunch'];
        $remaining_dinner    = isset($update_nas_router['selected_dinner'])    ? '' : $nasRecord['selected_dinner'];

        $all_refunded = empty($remaining_breakfast) && empty($remaining_lunch) && empty($remaining_dinner);

        if ($all_refunded) {
            $this->db->delete('nas_router', ['id' => $nasRecord['id']]);
        } else {
            $update_nas_router['total_amount'] = $nasRecord['total_amount'] - $refund_amount;
            $this->db->where('id', $nasRecord['id'])->update('nas_router', $update_nas_router);
        }

        echo json_encode(['status' => 'success', 'message' => 'Refund approved successfully.']);
    }
    public function check_staff_status()
    {
        $id = $this->input->post('id');

        $this->load->model('nas_model'); // single model used for both staff and assign_to

        $staff = $this->nas_model->get($id); // Get current staff info

        if (!$staff) {
            echo json_encode(['status' => 'inactive', 'message' => 'Staff not found.']);
            return;
        }

        if (strtolower($staff['status']) !== 'active') {
            echo json_encode([
                'status' => 'inactive',
                'message' => 'You are inactive. Please contact your branch admin.'
            ]);
            return;
        }

        if (!empty($staff['assign_to'])) {
            $manager = $this->nas_model->get($staff['assign_to']);
            if (!$manager || strtolower($manager['status']) !== 'active') {
                echo json_encode([
                    'status' => 'inactive',
                    'message' => 'Your Branch is Inactive. Please contact your Branch admin.'
                ]);
                return;
            }
        }

        echo json_encode(['status' => 'active']);
    }


    public function cancel_refund($id = '')
    {
        if (!get_permission('refund_client', 'is_edit')) {
            access_denied();
        }

        if (empty($id) || !is_numeric($id)) {
            echo json_encode(['status' => 'error', 'message' => 'Invalid refund ID.']);
            return;
        }

        $refund = $this->db->get_where('refund', ['id' => $id])->row_array();

        if (!$refund) {
            echo json_encode(['status' => 'error', 'message' => 'Refund record not found for ID: ' . $id]);
            return;
        }

        $this->db->where('id', $id)->update('refund', [
            'status' => 2,
            'cancelled_by' => get_loggedin_user_id(),
            'cancelled_at' => date('Y-m-d H:i:s')
        ]);

        echo json_encode(['status' => 'success', 'message' => 'Refund has been cancelled.']);
    }

    public function booking()
    {
        if (!get_permission('nas_booking', 'is_edit')) {
            access_denied();
        }

        $userID = get_loggedin_user_id();
        $RoleID = loggedin_role_id();

        // Filters
        $search_packageID = $this->input->post('package_list_id') ?: '';
        $search_userID = $this->input->post('reseller_subreseller_id') ?: $userID;
        $search_student_id = $this->input->post('student_id') ?: '';

        // Date Range
        $daterange = explode(' - ', $this->input->post('daterange'));
        if (!empty($daterange) && count($daterange) === 2) {
            $start = date("Y-m-d 00:00:00", strtotime($daterange[0]));
            $end = date("Y-m-d 23:59:59", strtotime($daterange[1]));
        } else {
            $start = date("Y-m-01 00:00:00");
            $end = date("Y-m-d 23:59:59");
        }

        // Set filters for view
        $this->data['filter_student_id'] = $search_student_id;
        $this->data['filter_packageID'] = $search_packageID;
        $this->data['filter_userID'] = $search_userID;
        $this->data['filter_daterange'] = $this->input->post('daterange');

        // Static view data
        $this->data['title'] = translate('Meal Booking');
        $this->data['package_list'] = $this->package_model->get_package_list_individual($RoleID, $userID);
        $this->data['zones_list'] = $this->zone_model->get_zone_list_individual($RoleID, $userID);
        $this->data['reseller_subreseller_list'] = $this->employee_model->get_reseller_subreseller_list($RoleID, $userID, '');
        $this->data['students'] = $this->nas_model->get_students_by_manager($userID);

        // Fetch raw NAS data
        $raw_naslist = $this->nas_model->get_nas_list_by_manager($RoleID, $search_userID, $search_packageID, $start, $end, $search_student_id);

        // Generate all dates between start and end range
        $all_dates = [];
        $current = strtotime(substr($start, 0, 10));
        $end_ts = strtotime(substr($end, 0, 10));
        while ($current <= $end_ts) {
            $all_dates[] = date('Y-m-d', $current);
            $current = strtotime('+1 day', $current);
        }

        $naslist = [];
        // Initialize mealCounts for all dates with 0
        $mealCounts = [
            'breakfast' => array_fill_keys($all_dates, 0),
            'lunch' => array_fill_keys($all_dates, 0),
            'dinner' => array_fill_keys($all_dates, 0),
        ];

        foreach ($raw_naslist as $row) {
            $student_key = $row['idd'];

            if (!isset($naslist[$student_key])) {
                $naslist[$student_key] = $row;
                $naslist[$student_key]['meals'] = [];
            }

            $selected_dates = !empty($row['selected_dates']) ? explode(',', $row['selected_dates']) : [];
            $selected_breakfast = !empty($row['selected_breakfast']) ? explode(',', $row['selected_breakfast']) : [];
            $selected_lunch = !empty($row['selected_lunch']) ? explode(',', $row['selected_lunch']) : [];
            $selected_dinner = !empty($row['selected_dinner']) ? explode(',', $row['selected_dinner']) : [];
            $package_names = !empty($row['package_name']) ? explode(',', $row['package_name']) : [];

            foreach ($selected_dates as $i => $date) {
                $date = trim($date);
                $package = isset($package_names[$i]) ? $package_names[$i] : '';

                if (!isset($naslist[$student_key]['meals'][$date])) {
                    // Initialize arrays for meals with package names
                    $naslist[$student_key]['meals'][$date] = [
                        'breakfast' => [],
                        'lunch' => [],
                        'dinner' => [],
                    ];
                }

                if (isset($selected_breakfast[$i]) && (int)$selected_breakfast[$i] === 1) {
                    $naslist[$student_key]['meals'][$date]['breakfast'][] = $package;
                    // Increase count for that date's breakfast
                    if (isset($mealCounts['breakfast'][$date])) {
                        $mealCounts['breakfast'][$date]++;
                    } else {
                        $mealCounts['breakfast'][$date] = 1;
                    }
                }
                if (isset($selected_lunch[$i]) && (int)$selected_lunch[$i] === 1) {
                    $naslist[$student_key]['meals'][$date]['lunch'][] = $package;
                    if (isset($mealCounts['lunch'][$date])) {
                        $mealCounts['lunch'][$date]++;
                    } else {
                        $mealCounts['lunch'][$date] = 1;
                    }
                }
                if (isset($selected_dinner[$i]) && (int)$selected_dinner[$i] === 1) {
                    $naslist[$student_key]['meals'][$date]['dinner'][] = $package;
                    if (isset($mealCounts['dinner'][$date])) {
                        $mealCounts['dinner'][$date]++;
                    } else {
                        $mealCounts['dinner'][$date] = 1;
                    }
                }
            }
        }

        $this->data['mealCounts'] = $mealCounts;
        $this->data['all_dates'] = $all_dates;
        $this->data['naslist'] = array_values($naslist);

        $this->data['sub_page'] = 'nas/booking';
        $this->data['main_menu'] = 'nas';
        $this->load->view('layout/index', $this->data);
    }

    public function add()
    {
        if (!get_permission('nas', 'is_add')) {
            access_denied();
        }

        $userID = get_loggedin_user_id();
        $RoleID = loggedin_role_id();

        $staff = null;
        if ($userID) {
            $staff = $this->db->get_where('staff', ['id' => $userID])->row();
        }

        if ($_POST) {
            $this->form_validation->set_rules('idd', 'Student ID', 'trim|required');
            $this->form_validation->set_rules('package_id', 'Package', 'trim|required');
            $this->form_validation->set_rules('student_name', 'Student Name', 'trim|required');
            $this->form_validation->set_rules('manager_name', 'Manager Name', 'trim|required');

            if ($this->form_validation->run() !== false) {
                $data = $this->input->post();
                $selectedDates = $data['selected_dates'] ?? [];
                $selectedBreakfast = $data['selected_breakfast'] ?? [];
                $selectedLunch = $data['selected_lunch'] ?? [];
                $selectedDinner = $data['selected_dinner'] ?? [];

                $count = count($selectedDates);
                $totalBookingAmount = 0;

                for ($i = 0; $i < $count; $i++) {
                    $mealTotal = 0;

                    if (!empty($selectedBreakfast[$i]) && $selectedBreakfast[$i] == 1) {
                        $mealTotal += floatval($data['breakfast_cost']);
                    }
                    if (!empty($selectedLunch[$i]) && $selectedLunch[$i] == 1) {
                        $mealTotal += floatval($data['lunch_cost']);
                    }
                    if (!empty($selectedDinner[$i]) && $selectedDinner[$i] == 1) {
                        $mealTotal += floatval($data['dinner_cost']);
                    }

                    $totalBookingAmount += $mealTotal;
                }

                if ($staff) {
                    if ($staff->user_balance >= $totalBookingAmount) {
                        $this->db->where('id', $userID)->update('staff', [
                            'user_balance' => $staff->user_balance - $totalBookingAmount
                        ]);

                        for ($i = 0; $i < $count; $i++) {
                            $mealTotal = 0;

                            if (!empty($selectedBreakfast[$i]) && $selectedBreakfast[$i] == 1) {
                                $mealTotal += floatval($data['breakfast_cost']);
                            }
                            if (!empty($selectedLunch[$i]) && $selectedLunch[$i] == 1) {
                                $mealTotal += floatval($data['lunch_cost']);
                            }
                            if (!empty($selectedDinner[$i]) && $selectedDinner[$i] == 1) {
                                $mealTotal += floatval($data['dinner_cost']);
                            }

                            if ($mealTotal == 0) {
                                continue;
                            }

                            $saveData = [
                                'idd' => $data['idd'],

                                'package_id' => $data['package_id'],
                                'package_name' => $data['package_name'],
                                'student_name' => $data['student_name'],
                                'manager_name' => $data['manager_name'],
                                'breakfast_cost' => $data['breakfast_cost'],
                                'lunch_cost' => $data['lunch_cost'],
                                'dinner_cost' => $data['dinner_cost'],
                                'total_breakfast' => $selectedBreakfast[$i] ?? 0,
                                'total_lunch' => $selectedLunch[$i] ?? 0,
                                'total_dinner' => $selectedDinner[$i] ?? 0,
                                'total_amount' => $mealTotal,
                                'selected_dates' => $data['selected_dates'],
                                'selected_breakfast' => $selectedBreakfast[$i] ?? 0,
                                'selected_lunch' => $selectedLunch[$i] ?? 0,
                                'selected_dinner' => $selectedDinner[$i] ?? 0,
                                'assign_to' => $userID
                            ];

                            $this->nas_model->nas_save($saveData);
                        }

                        set_alert('success', translate('Meal_Booking_has_been_saved_successfully'));
                    } else {
                        set_alert('error', translate('Insufficient_balance_to_book_the_meal'));
                    }
                } else {
                    set_alert('error', translate('User_not_found_or_invalid_userID'));
                }
            }
        }

        $this->data['packages'] = $this->package_model->get_package_list_individual($RoleID, $userID);
        $this->data['zones'] = ($userID == 1) ? $this->zone_model->findAll() : $this->zone_model->where('manager_id', $userID)->findAll();
        $this->data['students'] = $this->db->select('id, name')->get('staff')->result();
        $this->data['staff'] = $staff;
        $this->data['title'] = translate('Meal Booking');
        $this->data['sub_page'] = 'nas/add';
        $this->data['main_menu'] = 'nas';
        $this->load->view('layout/index', $this->data);
    }

    public function check_meal_ordered()
    {
        // Get input data from the client
        $data = json_decode(file_get_contents('php://input'), true);
        $date = $data['date'];
        $meal = $data['meal'];

        // Get logged-in user ID
        $user_id = get_loggedin_user_id(); // Use your helper method

        // Query to check existing meal order for this user and date
        $this->db->where('assign_to', $user_id);
        $this->db->where('selected_dates', $date);
        $existing_entries = $this->db->get('nas_router')->result();

        // If there are any existing entries for the date
        if ($existing_entries) {
            foreach ($existing_entries as $entry) {
                if ($meal === 'Breakfast' && $entry->selected_breakfast) {
                    echo json_encode(['status' => false, 'message' => 'Breakfast has already been ordered for this date.']);
                    return;
                }
                if ($meal === 'Lunch' && $entry->selected_lunch) {
                    echo json_encode(['status' => false, 'message' => 'Lunch has already been ordered for this date.']);
                    return;
                }
                if ($meal === 'Dinner' && $entry->selected_dinner) {
                    echo json_encode(['status' => false, 'message' => 'Dinner has already been ordered for this date.']);
                    return;
                }
            }
        }



        // No meal ordered yet
        echo json_encode(['status' => true]);
    }



    public function get_package_details()
    {
        $packageId = $this->input->post('package_id');

        if ($packageId) {
            $this->db->select('breakfast_cost, lunch_cost, dinner_cost');
            $this->db->from('packages');
            $this->db->where('id', $packageId);
            $query = $this->db->get();

            if ($query->num_rows() > 0) {
                $package = $query->row();
                echo json_encode([
                    'status' => true,
                    'breakfast_cost' => $package->breakfast_cost,
                    'lunch_cost' => $package->lunch_cost,
                    'dinner_cost' => $package->dinner_cost
                ]);
            } else {
                echo json_encode(['status' => false]);
            }
        }
    }

    public function get_students_by_branch()
    {
        $zone_id = $this->input->post('zone_id');
        if ($zone_id) {
            $students = $this->db->select('idd')
                ->where('zone_id', $zone_id)
                ->get('staff')
                ->result();

            echo json_encode(['status' => !empty($students), 'students' => $students]);
        } else {
            echo json_encode(['status' => false, 'students' => []]);
        }
    }

    public function get_student_name()
    {
        $idd = $this->input->post('idd');
        if ($idd) {
            $student = $this->db->select('name')->where('idd', $idd)->get('staff')->row();
            echo json_encode(['status' => !empty($student), 'student_name' => $student->name ?? '']);
        } else {
            echo json_encode(['status' => false, 'student_name' => '']);
        }
    }


    public function get_student_details()
    {
        $idd = $this->input->post('idd');
        if ($idd) {
            // Get student details from staff table
            $student = $this->db->select('name, assign_to')
                ->where('idd', $idd)
                ->get('staff')
                ->row();

            $manager_name = "";
            if (!empty($student->assign_to)) {
                // Get manager name using assign_to ID
                $manager = $this->db->select('name')
                    ->where('id', $student->assign_to)
                    ->get('staff')
                    ->row();
                $manager_name = $manager->name ?? '';
            }

            echo json_encode([
                'status' => !empty($student),
                'student_name' => $student->name ?? '',
                'manager_name' => $manager_name
            ]);
        } else {
            echo json_encode(['status' => false, 'student_name' => '', 'manager_name' => '']);
        }
    }
    public function get_students_by_zone()
    {
        $zoneId = $this->input->post('zone_id');

        // Query the database to get the students for the selected zone
        $students = $this->your_model->get_students_by_zone($zoneId);

        if ($students) {
            echo json_encode([
                'status' => 'success',
                'students' => $students
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'No students found'
            ]);
        }
    }


    public function edit($id = '')
    {
        // check access permission
        if (!get_permission('nas', 'is_edit')) {
            access_denied();
        }
        if ($this->input->post('submit') == 'update') {
            $data = $this->input->post();
            $this->nas_model->nas_save($data);
            set_alert('success', translate('information_has_been_updated_successfully'));
            redirect(base_url('nas'));
        }
        $this->data['nas_data'] = $this->nas_model->get_nas_details_by_id($id);
        $this->data['title'] = translate('Meal Booking');
        $this->data['sub_page'] = 'nas/edit';
        $this->data['main_menu'] = 'nas';
        $this->load->view('layout/index', $this->data);
    }

    public function application($id = '')
    {
        // check access permission
        if (!get_permission('nas', 'is_view')) {
            access_denied();
        }
        $this->db->select('id');
        $this->db->where('nas_id', $id);
        $this->db->from('staff');
        $query = $this->db->get();
        $count = $query->num_rows();
        if ($count > 0) {
            echo json_encode(['success' => true, 'count' => $count]);
        } else {
            echo json_encode(['success' => false]);
        }
    }



    // delete into appointment table by appointment id
    public function delete($id = '')
    {
        // check access permission
        if (!get_permission('nas', 'is_delete')) {
            access_denied();
        }
        $this->db->where('id', $id);
        $this->db->delete('nas_router');
    }

    // delete into appointment table by appointment id

    /**
     * @param int|null $id
     * @return object|string
     * @todo : add package to mikrotik
     */


    // add new product purchase bill
    public function purchase()
    {
        if (!get_permission('meal_order', 'is_view')) {
            access_denied();
        }

        $this->load->model('Nas_model'); // Load the model

        // Fetch all students
        $this->data['studentlist'] = $this->Nas_model->get_all_students();

        $this->data['purchaselist'] = $this->nas_model->get_purchase_list();
        $this->data['productlist'] = $this->nas_model->get_list('product', '', false, 'id,code,name');
        $this->data['supplierlist'] = $this->app_lib->getSelectList('supplier');
        $this->data['title'] = translate('Meal Order');
        $this->data['sub_page'] = 'nas/purchase';
        $this->data['main_menu'] = 'nas';

        $this->load->view('layout/index', $this->data);
    }


    public function purchase_save()
    {
        if (!get_permission('meal_order', 'is_add')) {
            access_denied();
        }
        if ($_POST) {
            // validate inputs
            $this->form_validation->set_rules('supplier_id', 'Supplier', 'trim|required');
            $this->form_validation->set_rules('bill_no', 'Bill No', 'trim|required');
            $this->form_validation->set_rules('buyer_name', 'Buyer Name', 'trim|required');
            $this->form_validation->set_rules('buyer_identity', 'Buyer Identity', 'trim|required');
            $this->form_validation->set_rules('buyer_mobile', 'Buyer Mobile', 'trim|required');
            $this->form_validation->set_rules('purchase_status', 'Purchase Status', 'trim|required');
            $this->form_validation->set_rules('date', 'Date', 'trim|required');
            $items = $this->input->post('purchases');
            foreach ($items as $key => $value) {
                $this->form_validation->set_rules('purchases[' . $key . '][product]', 'Category', 'trim|required');
                $this->form_validation->set_rules('purchases[' . $key . '][quantity]', 'Test Name', 'trim|required');
            }
            if ($this->form_validation->run() == false) {
                $msg = array(
                    'supplier_id' => form_error('supplier_id'),
                    'bill_no' => form_error('bill_no'),
                    'buyer_name' => form_error('buyer_name'),
                    'buyer_identity' => form_error('buyer_identity'),
                    'buyer_mobile' => form_error('buyer_mobile'),
                    'purchase_status' => form_error('purchase_status'),
                    'date' => form_error('date'),
                    'delivery_time' => form_error('delivery_time'),
                    'payment_amount' => form_error('payment_amount'),
                );
                foreach ($items as $key => $value) {
                    $msg['product' . $key] = form_error('purchases[' . $key . '][product]');
                    $msg['quantity' . $key] = form_error('purchases[' . $key . '][quantity]');
                }
                $array = array('status' => 'fail', 'url' => '', 'error' => $msg);
            } else {
                $data = $this->input->post();
                $this->nas_model->save_purchase($data);
                $url = base_url('nas/purchase');
                set_alert('success', translate('information_has_been_saved_successfully'));
                $array = array('status' => 'success', 'url' => $url, 'error' => '');
            }
            echo json_encode($array);
        }
    }



    // Update existing product purchase bill
    public function purchase_edit_save()
    {
        if (!get_permission('meal_order', 'is_edit')) {
            access_denied();
        }

        if ($_POST) {
            // Validate form inputs
            $this->form_validation->set_rules('supplier_id', 'Supplier', 'trim|required');
            $this->form_validation->set_rules('bill_no', 'Bill No', 'trim|required');
            $this->form_validation->set_rules('buyer_name', 'Buyer Name', 'trim|required');
            $this->form_validation->set_rules('buyer_identity', 'Buyer Identity', 'trim|required');
            $this->form_validation->set_rules('buyer_mobile', 'Buyer Mobile', 'trim|required');
            $this->form_validation->set_rules('purchase_status', 'Purchase Status', 'trim|required');
            $this->form_validation->set_rules('date', 'Date', 'trim|required');

            $items = $this->input->post('purchases');
            foreach ($items as $key => $value) {
                $this->form_validation->set_rules("purchases[$key][product]", 'Product', 'trim|required');
                $this->form_validation->set_rules("purchases[$key][quantity]", 'Quantity', 'trim|required|numeric');
            }

            if ($this->form_validation->run() == false) {
                $msg = array(
                    'supplier_id' => form_error('supplier_id'),
                    'bill_no' => form_error('bill_no'),
                    'buyer_name' => form_error('buyer_name'),
                    'buyer_identity' => form_error('buyer_identity'),
                    'buyer_mobile' => form_error('buyer_mobile'),
                    'purchase_status' => form_error('purchase_status'),
                    'date' => form_error('date'),
                );

                foreach ($items as $key => $value) {
                    $msg["product$key"] = form_error("purchases[$key][product]");
                    $msg["quantity$key"] = form_error("purchases[$key][quantity]");
                }

                echo json_encode(['status' => 'fail', 'error' => $msg]);
                return;
            }

            // Get form values
            $meal_bill_id = $this->input->post('meal_bill_id');
            $supplier_id = $this->input->post('supplier_id');
            $bill_no = $this->input->post('bill_no');
            $buyer_name = $this->input->post('buyer_name');
            $buyer_identity = $this->input->post('buyer_identity');
            $buyer_mobile = $this->input->post('buyer_mobile');
            $purchase_status = $this->input->post('purchase_status');
            $grand_total = $this->input->post('grand_total');
            $discount = $this->input->post('total_discount');
            $purchase_paid = $this->input->post('purchase_paid');
            $net_total = $this->input->post('net_grand_total');
            $date = $this->input->post('date');
            $remarks = $this->input->post('remarks');

            $payment_status = ($net_total <= $purchase_paid) ? 3 : 2;

            // Update meal_bill
            $array_invoice = array(
                'supplier_id' => $supplier_id,
                'bill_no' => $bill_no,
                'buyer_name' => $buyer_name,
                'buyer_identity' => $buyer_identity,
                'buyer_mobile' => $buyer_mobile,
                'remarks' => $remarks,
                'total' => $grand_total,
                'discount' => $discount,
                'due' => ($net_total - $purchase_paid),
                'purchase_status' => $purchase_status,
                'payment_status' => $payment_status,
                'hash' => app_generate_hash(),
                'date' => date('Y-m-d', strtotime($date)),
                'modifier_id' => get_loggedin_user_id(),
            );

            $this->db->where('id', $meal_bill_id);
            $this->db->update('meal_bill', $array_invoice);

            // Update meal_bill_details
            foreach ($items as $key => $value) {
                $array_product = array(
                    'meal_bill_id' => $meal_bill_id,
                    'product_id' => $value['product'],
                    'unit_price' => $value['unit_price'],
                    'discount' => $value['discount'],
                    'quantity' => $value['quantity'],
                    'sub_total' => $value['sub_total'],
                );

                if (isset($value['old_bill_details_id'])) {
                    $this->db->where('id', $value['old_bill_details_id']);
                    $this->db->update('meal_bill_details', $array_product);
                } else {
                    $this->db->insert('meal_bill_details', $array_product);
                }
            }

            // Redirect after success
            $url = base_url('nas/purchase/' . $meal_bill_id);
            set_alert('success', translate('information_has_been_updated_successfully'));

            echo json_encode(['status' => 'success', 'url' => $url]);
        }
    }

    public function purchase_edit($id)
    {
        if (!get_permission('meal_order', 'is_edit')) {
            access_denied();
        }

        $this->data['purchaselist'] = $this->nas_model->get_list('meal_bill', ['id' => $id], true);
        $this->data['productlist'] = $this->nas_model->get_list('product', "", false, 'id,code,name');
        $this->data['supplierlist'] = $this->app_lib->getSelectList('supplier');
        $this->data['title'] = translate('Meal Order');
        $this->data['sub_page'] = 'nas/purchase_edit';
        $this->data['main_menu'] = 'nas';

        $this->load->view('layout/index', $this->data);
    }



    // delete product purchase bill from database
    public function purchase_delete($id)
    {
        if (!get_permission('meal_order', 'is_delete')) {
            access_denied();
        }
        $this->db->where('id', $id);
        $this->db->delete('meal_bill');
        $this->db->where('meal_bill_id', $id);
        $this->db->delete('meal_bill_details');
        $this->db->where('meal_bill_id', $id);
        $this->db->delete('meal_payment_history');
    }

    public function purchase_bill($id = '', $hash = '')
    {
        if (!get_permission('meal_order', 'is_view')) {
            access_denied();
        }
        check_hash_restrictions('meal_bill', $id, $hash);
        $this->data['payvia_list'] = $this->app_lib->getSelectList('payment_method');
        $this->data['billdata'] = $this->nas_model->get_invoice($id);
        $this->data['title'] = translate('Meal Order');
        $this->data['sub_page'] = 'nas/purchase_bill';
        $this->data['main_menu'] = 'nas';
        $this->load->view('layout/index', $this->data);
    }

    // purchase partially payment add
    public function add_payment()
    {
        if (!get_permission('purchase_payment', 'is_add')) {
            access_denied();
        }
        if ($this->input->post()) {
            $data = $this->input->post();
            $data['getbill'] = $this->db->select('id,due,hash')->where('id', $data['meal_bill_id'])->get('meal_bill')->row_array();
            $this->form_validation->set_rules('paid_date', 'Paid Date', 'trim|required');
            $this->form_validation->set_rules('payment_amount', 'Payment Amount', 'trim|required|numeric|greater_than[1]|callback_payment_validation');
            $this->form_validation->set_rules('pay_via', 'Pay Via', 'trim|required');
            if ($this->form_validation->run() !== false) {
                $res = $this->nas_model->save_payment($data);
                if ($res['status'] == true) {
                    set_alert('success', translate('payment_successfull'));
                    if (get_permission('purchase_payment', 'is_view')) {
                        $this->session->set_flashdata('active_tab', 2);
                    }
                } else {
                    set_alert('error', $res['msg']);
                    $this->session->set_flashdata('active_tab', 1);
                }
                redirect(base_url('nas/purchase_bill/' . $data['meal_bill_id'] . '/' . $data['getbill']['hash']));
            } else {
                $this->session->set_flashdata('active_tab', 3);
                $this->meal_bill($data['meal_bill_id'], $data['getbill']['hash']);
            }
        }
    }

    // payment amount validation
    public function payment_validation($amount)
    {
        $bill_id = $this->input->post('meal_bill_id');
        $due_amount = $this->db->select('due')->where('id', $bill_id)->get('meal_bill')->row()->due;
        if ($amount <= $due_amount) {
            return true;
        } else {
            $this->form_validation->set_message("payment_validation", "Payment Amount Is More Than The Due Amount.");
            return false;
        }
    }

    public function unique_username($username)
    {
        $this->db->where('name', $username);
        $query = $this->db->get('nas_router');

        if ($query->num_rows() > 0) {
            $this->form_validation->set_message("unique_username", translate('nasname_has_already_been_used'));
            return false;
        } else {
            return true;
        }
    }
}
